/* 
 * Copyright (C) 2001-2003 Jacek Sieka, j_s@telia.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "stdinc.h"
#include "DCPlusPlus.h"

#include "DownloadManager.h"

#include "ConnectionManager.h"
#include "QueueManager.h"
#include "CryptoManager.h"
#include "HashManager.h"

#include "LogManager.h"
#include "SFVReader.h"
#include "User.h"
#include "File.h"
#include "FilteredFile.h"

#include "TraceManager.h"

#include "Client.h"


static const string DOWNLOAD_AREA = "Downloads";
const string Download::ANTI_FRAG_EXT = ".antifrag";

Download::Download(QueueItem* qi, User::Ptr& aUser) throw() : source(qi->getSourcePath(aUser)),
	target(qi->getTarget()), tempTarget(qi->getTempTarget()), file(NULL), isRollForward(0),
	crcCalc(NULL), bytesLeft(0), quickTick(GET_TICK()) { 
	
	setSize(qi->getSize());
	if(qi->isSet(QueueItem::FLAG_USER_LIST))
		setFlag(Download::FLAG_USER_LIST);
	if(qi->isSet(QueueItem::FLAG_MP3_INFO))
		setFlag(Download::FLAG_MP3_INFO);
	if(qi->isSet(QueueItem::FLAG_TESTSUR)) {
		setFlag(Download::FLAG_TESTSUR);
	} else {
	if(qi->isSet(QueueItem::FLAG_RESUME))
		setFlag(Download::FLAG_RESUME);
	if(qi->getCurrents()[0]->isSet(QueueItem::Source::FLAG_UTF8))
		setFlag(Download::FLAG_UTF8);
}
};

void DownloadManager::onTimerSecond(u_int32_t /*aTick*/) {
	Lock l(cs);

	Download::List tickList;
	iSpeed = SETTING(I_DOWN_SPEED);
	iHighSpeed = SETTING(H_DOWN_SPEED);
	iTime = SETTING(DOWN_TIME) * 60;
	throttleSetup();
	throttleZeroCounters();
	// Tick each ongoing download
	for(Download::Iter i = downloads.begin(); i != downloads.end(); ++i) {
        if((*i)->getStart() &&  0 == ((int)(GET_TICK() - (*i)->getStart()) / 1000 + 1) % 10 /*&& BOOLSETTING(AUTO_DROP_SOURCE)*/ ) // check every 10 sec
        {
            if((*i)->getRunningAverage() < 1230){
                QueueManager::getInstance()->autoDropSource((*i)->getUserConnection()->getUser());
                continue;
            }
        }

		if((*i)->getTotal() > 0) {
			tickList.push_back(*i);
		}
		if(BOOLSETTING(DISCONNECTING_ENABLE)) {
			if(getAverageSpeed() < (iHighSpeed*1024)) {
				Download* d = *i;
				dcassert(d->getUserConnection() != NULL);
				if (d->getSize() > (SETTING(MIN_FILE_SIZE) * (1024*1024))) {
					QueueItem* q = QueueManager::getInstance()->getRunning(d->getUserConnection()->getUser());
					if(d->getRunningAverage() < (iSpeed*1024) && (q->countOnlineUsers() >= 2) && (!d->isSet(Download::FLAG_USER_LIST))) {
						if(((GET_TICK() - d->quickTick)/1000) > iTime){
							d->getUserConnection()->disconnect();
			if(getAverageSpeed() < (SETTING(DISCONNECT)*1024)) { 
								QueueManager::getInstance()->removeSources(d->getUserConnection()->getUser(),QueueItem::Source::FLAG_SLOW);
							}
						}
					} else {
						d->quickTick = GET_TICK();
					}
				}
			}
		}
	}

	if(tickList.size() > 0)
		fire(DownloadManagerListener::TICK, tickList);
}

void DownloadManager::FileMover::moveFile(const string& source, const string& target) {
	Lock l(cs);
	files.push_back(make_pair(source, target));
	if(!active) {
		active = true;
		start();
	}
}

int DownloadManager::FileMover::run() {
	for(;;) {
		FilePair next;
		{
			Lock l(cs);
			if(files.empty()) {
				active = false;
				return 0;
			}
			next = files.back();
			files.pop_back();
		}
		try {
			File::renameFile(next.first, next.second);
		} catch(const FileException&) {
			// Too bad...
		}
	}
}

void DownloadManager::removeConnection(UserConnection::Ptr aConn, bool reuse /* = false */, bool reconnect /* = false */) {
	dcassert(aConn->getDownload() == NULL);
	aConn->removeListener(this);
	ConnectionManager::getInstance()->putDownloadConnection(aConn, reuse, reconnect);
}

void DownloadManager::checkDownloads(UserConnection* aConn, bool reconn /*=false*/) {

	bool slotsFull = (SETTING(DOWNLOAD_SLOTS) != 0) && (getDownloads() >= SETTING(DOWNLOAD_SLOTS));
	bool speedFull = (SETTING(MAX_DOWNLOAD_SPEED) != 0) && (getAverageSpeed() >= (SETTING(MAX_DOWNLOAD_SPEED)*1024));
	if( slotsFull || speedFull ) {
		bool extraFull = (SETTING(DOWNLOAD_SLOTS) != 0) && (getDownloads() >= (SETTING(DOWNLOAD_SLOTS)+SETTING(EXTRA_DOWNLOAD_SLOTS)));
		if(extraFull || !QueueManager::getInstance()->hasDownload(aConn->getUser(), QueueItem::HIGHEST)) {
			removeConnection(aConn);
			return;
		}
	}
	
	// this happen when download finished, we need reconnect.	
	if(reconn && QueueManager::getInstance()->hasDownload(aConn->getUser())) {
		removeConnection(aConn, false, true);
		return;
	}

	Download* d = QueueManager::getInstance()->getDownload(aConn->getUser());
	
	if(d) {
		dcassert(aConn->getDownload() == NULL);
		d->setUserConnection(aConn);
		aConn->setDownload(d);
		aConn->setState(UserConnection::STATE_FILELENGTH);

		{
			Lock l(cs);
			downloads.push_back(d);
		}
		
/*		
	if(d->isSet(Download::FLAG_RESUME)) {
			dcassert(d->getSize() != -1);


			const string& target = (d->getTempTarget().empty() ? d->getTarget() : d->getTempTarget());
			int64_t start = File::getSize(target);

			// Only use antifrag if we don't have a previous non-antifrag part
			if( BOOLSETTING(ANTI_FRAG) && (start == -1) && (d->getSize() != -1) ) {
				int64_t aSize = File::getSize(target + Download::ANTI_FRAG_EXT);

				if(aSize == d->getSize())
					start = d->getPos();
				else
					start = 0;

				d->setFlag(Download::FLAG_ANTI_FRAG);
			}
		
			int rollback = 16;//SETTING(ROLLBACK);
			if(rollback > start) {
				d->setPos(0);
			} else {
				d->setPos(start - rollback);
				//d->setFlag(Download::FLAG_ROLLBACK);
			}
		} else {
			d->setPos(0);
		}*/


		if(d->isSet(Download::FLAG_USER_LIST)) {
			if(aConn->isSet(UserConnection::FLAG_SUPPORTS_XML_BZLIST)) {
				d->setSource("files.xml.bz2");
			} else if(aConn->isSet(UserConnection::FLAG_SUPPORTS_BZLIST)) {
				d->setSource("MyList.bz2");
			}
		}

		d->bytesLeft = d->getSize() - d->getPos();
		if (d->bytesLeft <= 0) {
			d->bytesLeft = d->getSize();
			d->setPos(0);
		}

		if(BOOLSETTING(COMPRESS_TRANSFERS) && (aConn->isSet(UserConnection::FLAG_SUPPORTS_GETZBLOCK) || aConn->isSet(UserConnection::FLAG_SUPPORTS_GETTESTZBLOCK)) && d->getSize() != -1  && !aConn->getUser()->isSet(User::FORCEZOFF)) {
			// This one, we'll download with a zblock download instead...
			d->setFlag(Download::FLAG_ZDOWNLOAD);

			aConn->getZBlock(d->getSource(), d->getPos(), d->bytesLeft, d->isSet(Download::FLAG_UTF8));
		} else if(d->isSet(Download::FLAG_UTF8)) {
			aConn->getBlock(d->getSource(), d->getPos(), d->bytesLeft, true);
		} else{
			aConn->get(d->getSource(), d->getPos());
		}
		aConn->getUser()->unsetFlag(User::FORCEZOFF);
		return;
	}

	// Connection not needed any more, return it to the ConnectionManager...
	removeConnection(aConn, false, true);
}

void DownloadManager::onSending(UserConnection* aSource, int64_t aBytes) {
	if(aSource->getState() != UserConnection::STATE_FILELENGTH) {
		dcdebug("DM::onFileLength Bad state, ignoring\n");
		return;
	}
	
	if(prepareFile(aSource, (aBytes == -1) ? -1 : aSource->getDownload()->getPos() + aBytes)) {
		aSource->setDataMode();
	}
}

void DownloadManager::onFileLength(UserConnection* aSource, const string& aFileLength) {
        User::Ptr user = aSource->getUser();
	if(aSource->getState() != UserConnection::STATE_FILELENGTH) {
		dcdebug("DM::onFileLength Bad state, ignoring\n");
		return;
	}

	int64_t fileLength = Util::toInt64(aFileLength);
	if ( aSource->getDownload()->isSet(Download::FLAG_USER_LIST) ) {
		if((fileLength < 100) && (user != (User::Ptr)NULL) && (user->getBytesShared() > 0))
		{
//			user->setFileListTooSmall(true);
			user->setCheatingString(Util::validateMessage("Too small filelist - " + Util::formatBytes(fileLength) + " for the specified share of " + Util::formatBytes(user->getBytesShared()), false));
			user->setFakeSharing(true);
			User::updated(user);
			user->getClient()->fire(ClientListener::CHEAT_MESSAGE, user->getClient(), user->getNick()+": "+user->getCheatingString());

		}

		aSource->getUser()->setFileListSize(fileLength);
		(aSource->getUser()->getClient())->updated(aSource->getUser());
	}

	if(prepareFile(aSource, fileLength)) {
		aSource->setDataMode();
		aSource->startSend();
	}
}

template<bool managed>
class RollbackOutputStream : public OutputStream {
public:
	RollbackOutputStream(File* f, OutputStream* aStream, size_t bytes) : s(aStream), pos(0), bufSize(bytes), buf(new u_int8_t[bytes]) {
		size_t n = bytes;
		f->read(buf, n);
		f->movePos(-((int64_t)bytes));
	}
	virtual ~RollbackOutputStream() { if(managed) delete s; };

	virtual size_t flush() throw(FileException) {
		return s->flush();
	}

	virtual size_t write(const void* b, size_t len) throw(FileException) {
		u_int8_t* wb = (u_int8_t*)b;
		if(buf != NULL) {
			size_t n = len < (bufSize - pos) ? len : bufSize - pos;

	//		if(memcmp(buf + pos, wb, n) != 0) {
	//			throw FileException(STRING(ROLLBACK_INCONSISTENCY));
	//		}
			pos += n;
			if(pos == bufSize) {
				delete buf;
				buf = NULL;
			}
		}
		return s->write(wb, len);
	}

private:
	OutputStream* s;
	size_t pos;
	size_t bufSize;
	u_int8_t* buf;
};

template<bool managed>
class TigerTreeOutputStream : public OutputStream {
public:
	TigerTreeOutputStream(const TigerTree& aTree, OutputStream* aStream) : s(aStream), real(aTree), cur(aTree.getBlockSize()), verified(0) {
	}
	virtual ~TigerTreeOutputStream() { if(managed) delete s; };

	virtual size_t flush() throw(FileException) {
		cur.finalize();
		checkTrees();
		return s->flush();
	}

	virtual size_t write(const void* b, size_t len) throw(FileException) {
		cur.update(b, len);
		checkTrees();
		return s->write(b, len);
	}
	
	virtual int64_t verifiedBytes() {
		return cur.getFileSize();
	}
private:
	OutputStream* s;
	TigerTree real;
	TigerTree cur;
	size_t verified;

	void checkTrees() throw(FileException) {
		while(cur.getLeaves().size() > verified) {
			if(cur.getLeaves().size() > real.getLeaves().size() ||
				!(cur.getLeaves()[verified] == real.getLeaves()[verified])) 
			{
				throw FileException(STRING(TTH_INCONSISTENCY));
			}
			verified++;
		}
	}
};

bool DownloadManager::prepareFile(UserConnection* aSource, int64_t newSize /* = -1 */) {
	Download* d = aSource->getDownload();
	dcassert(d != NULL);

	if(newSize != -1)
		d->setSize(newSize);

	if(d->getPos() >= d->getSize()) {
		// Already finished?
		aSource->setDownload(NULL);
		removeDownload(d, true);
		removeConnection(aSource);
		return false;
	}

	dcassert(d->getSize() != -1);

	string target = d->getDownloadTarget();
	Util::ensureDirectory(target);
	if(d->isSet(Download::FLAG_USER_LIST)) {
		if(aSource->isSet(UserConnection::FLAG_SUPPORTS_XML_BZLIST)) {
			target += ".xml.bz2";
		} else if(aSource->isSet(UserConnection::FLAG_SUPPORTS_BZLIST)) {
			target += ".bz2";
		} else {
			target += ".DcLst";
		}
	}

	File* file = NULL;
	try {
		// Let's check if we can find this file in a any .SFV...
		int trunc = d->isSet(Download::FLAG_RESUME) ? 0 : File::TRUNCATE;
		file = new File(target, File::RW, File::OPEN | File::CREATE | trunc);
/*		
		if(d->isSet(Download::FLAG_ANTI_FRAG)) {
						file->setSize(d->getSize());
*/
//				if(!d->isSet(Download::FLAG_USER_LIST) && BOOLSETTING(MEMORY_MAPPED_FILE)){
//			file = new MappedFile(target, d->getSize(), File::RW, File::OPEN | File::CREATE | trunc, sfvcheck);			
//		} else {
//			file = new BufferedFile(target, File::RW, File::OPEN | File::CREATE | trunc, sfvcheck);			
		//}
		file->setPos(d->getPos());		
	} catch(const Exception& e) {
		delete file;
		fire(DownloadManagerListener::FAILED, d, e.getError());
		aSource->setDownload(NULL);
		removeDownload(d);
		removeConnection(aSource);
		return false;
	}

	d->setFile(file);
	
//	if(d->isSet(Download::FLAG_ROLLBACK)) {
//		d->setFile(new RollbackOutputStream<true>(file, d->getFile(), SETTING(ROLLBACK)/*(size_t)min((int64_t)SETTING(ROLLBACK), d->getSize() - d->getPos())*/));
//	}
	
	bool sfvcheck = BOOLSETTING(SFV_CHECK) && (d->getPos() == 0) && (SFVReader(d->getTarget()).hasCRC());

	if(sfvcheck) {
		d->setFlag(Download::FLAG_CALC_CRC32);
		Download::CrcOS* crc = new Download::CrcOS(d->getFile());
		d->setCrcCalc(crc);
		d->setFile(crc);
	}
	
//	TigerTree tree;
//	if(HashManager::getInstance()->getTree(d->getDownloadTarget(), tree)) {
//		d->setFile(new TigerTreeOutputStream<true>(tree, d->getFile()));
//	}

	if(d->isSet(Download::FLAG_ZDOWNLOAD)) {
		d->setFile(new FilteredOutputStream<UnZFilter, true>(d->getFile()));
	}
	dcassert(d->getPos() != -1);
	d->setStart(GET_TICK());
	aSource->setState(UserConnection::STATE_DONE);
	throttleSetup();	
	
	fire(DownloadManagerListener::STARTING, d);
	
	return true;
}	

void DownloadManager::onData(UserConnection* aSource, const u_int8_t* aData, int aLen) {
	Download* d = aSource->getDownload();
	dcassert(d != NULL);
	
	throttleBytesTransferred(aLen);
				try {

	int64_t velikost = d->getFile()->write(aData, aLen);
	d->addPos(velikost);
	d->addActual(aLen);
	

	if((d->getPos() == d->getSize()) && (d->isSet(Download::FLAG_USER_LIST))) {
			handleEndData(aSource);
			aSource->setLineMode();
			return;
		}

	if(d->getQueueTotal() == d->getSize()) {
			handleEndData(aSource);
			aSource->setLineMode();
			return;
	}

	} catch(const BlockDLException) {
		fire(DownloadManagerListener::FAILED, d, CSTRING(BLOCK_FINISHED));
		aSource->setDownload(NULL);
		removeDownload(d);
		removeConnection(aSource, false, true);
	} catch(const FileDLException) {
		handleEndData(aSource);
	} catch(const FileException& e) {
		fire(DownloadManagerListener::FAILED, d, e.getError());
		
		d->setPos(d->getPos() - d->getTotal());
		aSource->setDownload(NULL);
		removeDownload(d);
		removeConnection(aSource);
	} catch(const Exception& e) {
		fire(DownloadManagerListener::FAILED, d, e.getError());
		// Nuke the bytes we have written, this is probably a compression error
		d->setPos(d->getPos() - d->getTotal());
		aSource->setDownload(NULL);
		removeDownload(d);
		removeConnection(aSource);
	}
}

void DownloadManager::onModeChange(UserConnection* aSource, int /*aNewMode*/) {
	
	Download* d = aSource->getDownload();
	dcassert(d != NULL);

	if(d->isSet(Download::FLAG_USER_LIST) )
	handleEndData(aSource);
}

/** Download finished! */
void DownloadManager::handleEndData(UserConnection* aSource) {
	dcassert(aSource->getState() == UserConnection::STATE_DONE);
	Download* d = aSource->getDownload();
	dcassert(d != NULL);

	u_int32_t crc = 0;
	bool hasCrc = (d->getCrcCalc() != NULL);
	// First, finish writing the file (flushing the buffers and closing the file...)
	try {
	//	d->getFile()->flush();
		if(hasCrc)
			crc = d->getCrcCalc()->getFilter().getValue();
		delete d->getFile();
		d->setFile(NULL);
		d->setCrcCalc(NULL);
		// Check if we're anti-fragging...
	/*	if(d->isSet(Download::FLAG_ANTI_FRAG)) {
			// Ok, rename the file to what we expect it to be...
			try {
				const string& tgt = d->getTempTarget().empty() ? d->getTarget() : d->getTempTarget();
				File::renameFile(d->getDownloadTarget(), tgt);
				d->unsetFlag(Download::FLAG_ANTI_FRAG);
			} catch(const FileException& e) {
				dcdebug("AntiFrag: %s\n", e.getError().c_str());
				// Now what?
			}
		}*/
	} catch(const FileException& e) {
		fire(DownloadManagerListener::FAILED, d, e.getError());
		
		aSource->setDownload(NULL);
		removeDownload(d);
		removeConnection(aSource, false, true);
		return;
	}

	//dcassert(d->getPos() == d->getSize());
	dcdebug("Download finished: %s, size " I64_FMT ", downloaded " I64_FMT "\n", d->getTarget().c_str(), d->getSize(), d->getTotal());

	// Check if we have some crc:s...
	if(BOOLSETTING(SFV_CHECK)) {
		SFVReader sfv(d->getTarget());
		if(sfv.hasCRC()) {
			bool crcMatch;
			string tgt = d->getDownloadTarget();
			if(hasCrc) {
				crcMatch = (crc == sfv.getCRC());
			} else {
				// More complicated, we have to reread the file
				try {
					
					File ff(tgt, File::READ, File::OPEN);
					CalcInputStream<CRC32Filter, false> f(&ff);

					const size_t BUF_SIZE = 16 * 65536;
					AutoArray<u_int8_t> b(BUF_SIZE);
					size_t n = BUF_SIZE;
					while(f.read((u_int8_t*)b, n) > 0)
						;		// Keep on looping...

					crcMatch = (f.getFilter().getValue() == sfv.getCRC());
				} catch (FileException&) {
					// Nope; read failed...
					goto noCRC;
				}
			}
			if(!crcMatch) {
				File::deleteFile(tgt);
		//		dcdebug("DownloadManager: CRC32 mismatch for %s\n", d->getTarget().c_str());
				LogManager::getInstance()->message(STRING(SFV_INCONSISTENCY) + " (" + STRING(FILE) + ": " + d->getTarget() + ")");

				fire(DownloadManagerListener::FAILED, d, STRING(SFV_INCONSISTENCY));
				
				string target = d->getTarget();
				
				aSource->setDownload(NULL);
				removeDownload(d);				
				
				QueueManager::getInstance()->removeSource(target, aSource->getUser(), QueueItem::Source::FLAG_CRC_WARN, false);
				checkDownloads(aSource, true);
				return;
			} 

			d->setFlag(Download::FLAG_CRC32_OK);
			
			dcdebug("DownloadManager: CRC32 match for %s\n", d->getTarget().c_str());
		}
	}
noCRC:


	if(BOOLSETTING(LOG_DOWNLOADS) && (BOOLSETTING(LOG_FILELIST_TRANSFERS) || !d->isSet(Download::FLAG_USER_LIST))) {
		StringMap params;
		params["target"] = d->getTarget();
		params["user"] = aSource->getUser()->getNick();
		params["hub"] = aSource->getUser()->getLastHubName();
		params["hubip"] = aSource->getUser()->getLastHubAddress();
		params["size"] = Util::toString(d->getSize());
		params["sizeshort"] = Util::formatBytes(d->getSize());
		params["chunksize"] = Util::toString(d->getTotal());
		params["chunksizeshort"] = Util::formatBytes(d->getTotal());
		params["actualsize"] = Util::toString(d->getActual());
		params["actualsizeshort"] = Util::formatBytes(d->getActual());
		params["speed"] = Util::formatBytes(d->getAverageSpeed()) + "/s";
		params["time"] = Util::formatSeconds((GET_TICK() - d->getStart()) / 1000);
		params["sfv"] = Util::toString(d->isSet(Download::FLAG_CRC32_OK) ? 1 : 0);
		LOG(DOWNLOAD_AREA, Util::formatParams(SETTING(LOG_FORMAT_POST_DOWNLOAD), params));
	}
	
	// Stop all segments
	for(Download::Iter i = downloads.begin(); i != downloads.end(); ++i) {
		Download* d1 = *i;
		if(d1->getTarget() == d->getTarget()) {
			if(d1 != d)
			{
			d1->getUserConnection()->disconnect();
			d1->getUserConnection()->setDownload(NULL);
			}
		}
	}
	
	// Check hash
	if((BOOLSETTING(CHECK_TTH)) && (!d->isSet(Download::FLAG_USER_LIST)) && (!d->isSet(Download::FLAG_MP3_INFO)))
	{
		QueueItem* q = QueueManager::getInstance()->fileQueue.find(d->getTarget());
		TTHValue* hash1 = new TTHValue(HashManager::getInstance()->getTTfromFile(d->getTempTarget()).getRoot());
		TTHValue* hash2 = q->getTTH();

		bool hashMatch = true;

		if((hash1 != NULL) && (hash2 != NULL)) 
		{ 
		
		if (*hash1 == *hash2) hashMatch = true; else hashMatch = false;

		if(!hashMatch) {		
		//	File::deleteFile(d->getDownloadTarget());
			LogManager::getInstance()->message(STRING(TTH_INCONSISTENCY) + " (" + d->getTarget() + ")");

			fire(DownloadManagerListener::FAILED, d, STRING(TTH_INCONSISTENCY));
				
			string target = d->getTarget();
				
			aSource->setDownload(NULL);
			d->setPos(0);
			removeDownload(d);				
			checkDownloads(aSource, true);
			return;
		}
	
		d->setFlag(Download::FLAG_TTH_OK);

		}

		delete hash1;

	}


	// Check if we need to move the file
	if( !d->getTempTarget().empty() && (Util::stricmp(d->getTarget().c_str(), d->getTempTarget().c_str()) != 0) ) {
		try {
			Util::ensureDirectory(d->getTarget());
			if(File::getSize(d->getTempTarget()) > MOVER_LIMIT) {
				mover.moveFile(d->getTempTarget(), d->getTarget());
			} else {
			File::renameFile(d->getTempTarget(), d->getTarget());
			}
			d->setTempTarget(Util::emptyString);
		} catch(const FileException&) {
			// Huh??? Now what??? Oh well...let it be...
		}
	}
	fire(DownloadManagerListener::COMPLETE, d);

//	abortDownload(d->getTarget());
	aSource->setDownload(NULL);
	removeDownload(d, true);	
	checkDownloads(aSource, true);

}

void DownloadManager::onMaxedOut(UserConnection* aSource) { 
	if(aSource->getState() != UserConnection::STATE_FILELENGTH) {
		dcdebug("DM::onMaxedOut Bad state, ignoring\n");
		return;
	}

	Download* d = aSource->getDownload();
	dcassert(d != NULL);

	fire(DownloadManagerListener::FAILED, d, STRING(NO_SLOTS_AVAILABLE));
	// CDM EXTENSION BEGINS
	User::Ptr user = aSource->getUser();
	//if(user != NULL)
	//{	
		//Client::Ptr client = user->getClient();
		//if(client != NULL)
		//{
			if( d->isSet(Download::FLAG_TESTSUR) && user->getConnection().size() > 0) {
				dcdebug("No slots for TestSUR %s\n", user->getNick());
				//user->unsetTSURFlags();
				//user->setFlag(User::TSUR_NO_SLOTS_AVAILABLE);
				user->setTestSUR("MaxedOut");
				user->setHasTestSURinQueue(false);
				user->updateClientType();
				//user->setTestSURComplete(true);
				aSource->setDownload(NULL);
				removeDownload(d, true);
				removeConnection(aSource);
				user->setCheatingString(Util::validateMessage("No slots for TestSUR. User is using slotlocker.", false));
				user->setFakeSharing(true);
				User::updated(user);
				user->getClient()->fire(ClientListener::CHEAT_MESSAGE, user->getClient(), user->getNick()+": "+user->getCheatingString());
				return;
			}
			/*else if((d->isSet(Download::FLAG_USER_LIST)) && (client->getOp()) 
				&& (user->isSet(User::DCPLUSPLUS)) && ( client->getClientFavNoSlotsDcpp() ))
			{
				user->setFileListNoFreeSlots(true);
				user->setCheatingString(STRING(CHEATING_FILELIST_NO_FREE_SLOTS));
				//user->sendFakeShareCommand();
				string rawCommand = client->getClientKickNoSlotsForFilelistString();
				user->sendRawCommand(rawCommand, client);
				if(!BOOLSETTING(SUPPRESS_CHEATING_MESSAGES))
				{
					client->addLine("*** User " + user->getNick() + " " + user->getCheatingString());
				}
				client->updated(user);
			}*/
		//}
	//}
	// CDM EXTENSION ENDS

	aSource->setDownload(NULL);
	removeDownload(d);
	removeConnection(aSource);
}

void DownloadManager::onFailed(UserConnection* aSource, const string& aError) {
	Download* d = aSource->getDownload();

	if(d == NULL) {
		removeConnection(aSource);
		return;
	}
	
	fire(DownloadManagerListener::FAILED, d, aError);
	// CDM EXTENSION BEGINS
	if( d->isSet(Download::FLAG_TESTSUR) ) {
		dcdebug("TestSUR Error: %s\n", aError);
		User::Ptr user = aSource->getUser();
		user->setTestSUR(aError);
		user->setHasTestSURinQueue(false);
		user->updateClientType();
		aSource->setDownload(NULL);
		removeDownload(d, true);
		removeConnection(aSource);
		return;
	}
	// CDM EXTENSION ENDS

	string target = d->getTarget();
	aSource->setDownload(NULL);
	removeDownload(d);
	
	if(aError.find("File Not Available") != string::npos) {
		QueueManager::getInstance()->removeSource(target, aSource->getUser(), QueueItem::Source::FLAG_FILE_NOT_AVAILABLE, false);
	}

	removeConnection(aSource);
}

void DownloadManager::removeDownload(Download* d, bool finished /* = false */) {
	if(d->getFile()) {
		try {
			d->getFile()->flush();
		} catch(const Exception&) {
			finished = false;
		}
		delete d->getFile();
		d->setFile(NULL);
		d->setCrcCalc(NULL);

		if(d->isSet(Download::FLAG_ANTI_FRAG)) {
			// Ok, set the pos to whereever it was last writing and hope for the best...
			d->unsetFlag(Download::FLAG_ANTI_FRAG);
		} 
	}

	{
		Lock l(cs);
		// Either I'm stupid or the msvc7 optimizer is doing something _very_ strange here...
		// STL-port -D_STL_DEBUG complains that .begin() and .end() don't have the same owner (!),
		// but only in release build

		dcassert(find(downloads.begin(), downloads.end(), d) != downloads.end());

		//		downloads.erase(find(downloads.begin(), downloads.end(), d));
		
		for(Download::Iter i = downloads.begin(); i != downloads.end(); ++i) {
			if(*i == d) {
				downloads.erase(i);
				break;
			}
		}
	}
	QueueManager::getInstance()->putDownload(d, finished);
}

void DownloadManager::abortDownload(const string& aTarget) {
	Lock l(cs);
	for(Download::Iter i = downloads.begin(); i != downloads.end(); ++i) {
		Download* d = *i;
		if(d->getTarget() == aTarget) {
			dcassert(d->getUserConnection() != NULL);
			d->getUserConnection()->disconnect();
		}
	}
}

void DownloadManager::abortDownload(const string& aTarget, User::Ptr& aUser) {
	Lock l(cs);
	for(Download::Iter i = downloads.begin(); i != downloads.end(); ++i) {
		Download* d = *i;
		if(d->getTarget() == aTarget) {
			dcassert(d->getUserConnection() != NULL);
			if(d->getUserConnection()->getUser() == aUser){
				d->getUserConnection()->disconnect();
		
						break;
		}
	
	}
	}
}

void DownloadManager::onFileNotAvailable(UserConnection* aSource) throw() {
	Download* d = aSource->getDownload();
	dcassert(d != NULL);

	dcdebug("File Not Available: %s\n", d->getTarget().c_str());

	if(d->getFile()) {
		delete d->getFile();
		d->setFile(NULL);
		d->setCrcCalc(NULL);
	}

	fire(DownloadManagerListener::FAILED, d, d->getTargetFileName() + ": " + STRING(FILE_NOT_AVAILABLE));
	// CDM EXTENSION BEGINS
	if( d->isSet(Download::FLAG_TESTSUR) ) {
		dcdebug("TestSUR File not available\n");
		User::Ptr user = aSource->getUser();
		user->setTestSUR("File not available");
		user->setHasTestSURinQueue(false);
		user->updateClientType();
		aSource->setDownload(NULL);
		removeDownload(d, true);
		removeConnection(aSource);
		return;
	}
	// CDM EXTENSION ENDS

	aSource->setDownload(NULL);

	QueueManager::getInstance()->removeSource(d->getTarget(), aSource->getUser(), QueueItem::Source::FLAG_FILE_NOT_AVAILABLE, false);
	removeDownload(d, false);
	checkDownloads(aSource);
}

// UserConnectionListener
void DownloadManager::onAction(UserConnectionListener::Types type, UserConnection* conn) throw() {
	switch(type) {
	case UserConnectionListener::MAXED_OUT: onMaxedOut(conn); break;
	case UserConnectionListener::FILE_NOT_AVAILABLE: onFileNotAvailable(conn); break;
	default: break;
	}
}
void DownloadManager::onAction(UserConnectionListener::Types type, UserConnection* conn, const string& line) throw() {
	switch(type) {
	case UserConnectionListener::FILE_LENGTH:
		onFileLength(conn, line); break;
	case UserConnectionListener::FAILED:
		onFailed(conn, line); break;
	default:
		break;
	}
}
void DownloadManager::onAction(UserConnectionListener::Types type, UserConnection* conn, int64_t bytes) throw() {
	if(type == UserConnectionListener::SENDING)
		onSending(conn, bytes);
}
void DownloadManager::onAction(UserConnectionListener::Types type, UserConnection* conn, const u_int8_t* data, int len) throw() {
	switch(type) {
	case UserConnectionListener::DATA:
		onData(conn, data, len); break;
	default:
		break;
	}
}

void DownloadManager::onAction(UserConnectionListener::Types type, UserConnection* conn, int mode) throw() {
	switch(type) {
	case UserConnectionListener::MODE_CHANGE:
		onModeChange(conn, mode); break;
	default:
		break;
	}
}

// TimerManagerListener
void DownloadManager::onAction(TimerManagerListener::Types type, u_int32_t aTick) throw() {
	switch(type) {
	case TimerManagerListener::SECOND:
		onTimerSecond(aTick); break;
	default:
		break;
	}
}

void DownloadManager::throttleReturnBytes(u_int32_t b) {
	Lock l(cs);
#ifdef _DEBUG
	dcdebug("DownloadManager::throttleReturnBytes - Called with %d\n", b);
#endif
	if (b > 0 && b < 2*mByteSlice) {
		mBytesSpokenFor -= b;
		if (mBytesSpokenFor < 0)
			mBytesSpokenFor = 0;
	}
}

int32_t DownloadManager::throttleGetSlice() {
	if (mThrottleEnable) {
		Lock l(cs);
		u_int32_t left = mDownloadLimit - mBytesSpokenFor;
#ifdef _DEBUG
		//dcdebug("DownloadManager::throttleGetSlice - Limit: %d  Read: %d  Left: %d\n", mDownloadLimit, mBytesSent, left);
#endif
		if (left > 0) {
			if (left > 2*mByteSlice) {
				mBytesSpokenFor += mByteSlice;
				return mByteSlice;
			} else {
				mBytesSpokenFor += left;
				return left;
			}
		} else
			return 0;
	} else
		return -1;
}

u_int32_t DownloadManager::throttleCycleTime() {
	if (mThrottleEnable)
		return mCycleTime;
	return 0;
}

void DownloadManager::throttleZeroCounters() {
	Lock l(cs);
	mBytesSpokenFor = 0;
	mBytesSent = 0;
}

void DownloadManager::throttleBytesTransferred(u_int32_t i) {
	Lock l(cs);
	mBytesSent += i;
}

void DownloadManager::throttleSetup() {
	// called once a second, plus when a download starts
#define INBUFSIZE 64*1024
	Lock l(cs);
	unsigned int num_transfers = getDownloads();
	mDownloadLimit = (SETTING(MAX_DOWNLOAD_SPEED_LIMIT) * 1024);
	mThrottleEnable = BOOLSETTING(THROTTLE_ENABLE) && (mDownloadLimit > 0) && (num_transfers > 0);
	if (mThrottleEnable) {
		if (num_transfers > 0) {
			if (mDownloadLimit <= (INBUFSIZE * 10 * num_transfers)) {
				mByteSlice = mDownloadLimit / (7 * num_transfers);
				if (mByteSlice > INBUFSIZE)
					mByteSlice = INBUFSIZE;
				mCycleTime = 1000 / 10;
				}
			else {
				mByteSlice = INBUFSIZE;
				mCycleTime = 1000 * INBUFSIZE / mDownloadLimit;
			}
		}
#ifdef _DEBUG
		dcdebug("DownloadManager::throttleSetup - byte slice: %d  cycle time: %d\n", mByteSlice, mCycleTime);
#endif
	}
}

/**
 * @file
 * $Id: DownloadManager.cpp,v 1.96 2004/03/09 21:40:49 arnetheduck Exp $
 */
