/* 
 * Copyright (C) 2001-2003 Jacek Sieka, j_s@telia.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#if !defined(AFX_HUBMANAGER_H__75858D5D_F12F_40D0_B127_5DDED226C098__INCLUDED_)
#define AFX_HUBMANAGER_H__75858D5D_F12F_40D0_B127_5DDED226C098__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "SettingsManager.h"

#include "CriticalSection.h"
#include "HttpConnection.h"
#include "User.h"
#include "UserCommand.h"
#include "FavoriteUser.h"
#include "Singleton.h"
// CDM EXTENSION BEGINS (profiles)
class ClientProfile {
	public:
	typedef vector<ClientProfile> List;
	typedef List::iterator Iter;

	ClientProfile() : id(0), priority(0), rawToSend(0) { };

	ClientProfile(int aId, const string& aName, const string& aVersion, const string& aTag, 
		const string& aExtendedTag, const string& aLock, const string& aPk, const string& aSupports,
		const string& aTestSUR, const string& aUserConCom, const string& aStatus, const string& aCheatingDescription, 
		int aRawToSend, int aTagVersion, int aUseExtraVersion, int aCheckMismatch) 
		throw() : id(aId), name(aName), version(aVersion), tag(aTag), extendedTag(aExtendedTag), 
		lock(aLock), pk(aPk), supports(aSupports), testSUR(aTestSUR), userConCom(aUserConCom),  status(aStatus),
		cheatingDescription(aCheatingDescription), rawToSend(aRawToSend), tagVersion(aTagVersion),
		useExtraVersion(aUseExtraVersion), checkMismatch(aCheckMismatch) { };

	ClientProfile(const ClientProfile& rhs) : id(rhs.id), name(rhs.name), version(rhs.version), tag(rhs.tag), 
		extendedTag(rhs.extendedTag), lock(rhs.lock), pk(rhs.pk), supports(rhs.supports), testSUR(rhs.testSUR), 
		userConCom(rhs.userConCom), status(rhs.status), cheatingDescription(rhs.cheatingDescription), rawToSend(rhs.rawToSend),
		tagVersion(rhs.tagVersion), useExtraVersion(rhs.useExtraVersion), checkMismatch(rhs.checkMismatch)
	{

	}

	ClientProfile& operator=(const ClientProfile& rhs) {
		id = rhs.id;
		name = rhs.name;
		version = rhs.version;
		tag = rhs.tag;
		extendedTag = rhs.extendedTag;
		lock = rhs.lock;
		pk = rhs.pk;
		supports = rhs.supports;
		testSUR = rhs.testSUR;
		userConCom = rhs.userConCom;
		status = rhs.status;
		rawToSend = rhs.rawToSend;
		cheatingDescription = rhs.cheatingDescription;
		tagVersion = rhs.tagVersion;
		useExtraVersion = rhs.useExtraVersion;
		checkMismatch = rhs.checkMismatch;
		return *this;
	}

		GETSET(int, id, Id);
		GETSETREF(string, name, Name);
		GETSETREF(string, version, Version);
		GETSETREF(string, tag, Tag);
		GETSETREF(string, extendedTag, ExtendedTag);
		GETSETREF(string, lock, Lock);
		GETSETREF(string, pk, Pk);
		GETSETREF(string, supports, Supports);
		GETSETREF(string, testSUR, TestSUR);
		GETSETREF(string, userConCom, UserConCom);
		GETSETREF(string, status, Status);
		GETSETREF(string, cheatingDescription, CheatingDescription);
		GETSET(int, priority, Priority);
		GETSET(int, rawToSend, RawToSend);
		GETSET(int, tagVersion, TagVersion);
		GETSET(int, useExtraVersion, UseExtraVersion);
		GETSET(int, checkMismatch, CheckMismatch);
};
// CDM EXTENSION ENDS

class HubEntry {
public:
	typedef vector<HubEntry> List;
	typedef List::iterator Iter;
	
	// XML stuff
	HubEntry(const string& aName, const string& aServer, const string& aDescription, const string& aUsers, 
		const string& aShared, const string& aCountry, const string& aStatus, const string& aMinshare, const string& aMinslots, 
		const string& aMaxhubs, const string& aMaxusers, const string& aReliability, const string& aRating, const string& aPort) throw() : 
	name(aName), server(aServer), description(aDescription), users(aUsers), 
		shared(aShared), country(aCountry), status(aStatus), minshare(aMinshare), minslots(aMinslots), 
		maxhubs(aMaxhubs), maxusers(aMaxusers), reliability(aReliability), rating(aRating), port(aPort) { };
	//
	HubEntry(const string& aName, const string& aServer, const string& aDescription, const string& aUsers) throw() : 
	name(aName), server(aServer), description(aDescription), users(aUsers) { };
	HubEntry() throw() { };
	HubEntry(const HubEntry& rhs) throw() : name(rhs.name), server(rhs.server), description(rhs.description), users(rhs.users), 
		shared(rhs.shared), country(rhs.country), status(rhs.status), minshare(rhs.minshare), minslots(rhs.minslots), 
		maxhubs(rhs.maxhubs), maxusers(rhs.maxusers), reliability(rhs.reliability), rating(rhs.rating), port(rhs.port) { }
	virtual ~HubEntry() throw() { };

	GETSETREF(string, name, Name);
	GETSETREF(string, server, Server);
	GETSETREF(string, description, Description);
	GETSETREF(string, users, Users);
	// XML stuff
	GETSETREF(string, shared, Shared);
	GETSETREF(string, country, Country);
	GETSETREF(string, status, Status);
	GETSETREF(string, minshare, Minshare);
	GETSETREF(string, minslots, Minslots);
	GETSETREF(string, maxhubs, Maxhubs);
	GETSETREF(string, maxusers, Maxusers);
	GETSETREF(string, reliability, Reliability);
	GETSETREF(string, rating, Rating);
	GETSETREF(string, port, Port);
	//
};

class FavoriteHubEntry {
public:
	typedef FavoriteHubEntry* Ptr;
	typedef vector<Ptr> List;
	typedef List::iterator Iter;

	FavoriteHubEntry() throw() : connect(false), windowposx(0), windowposy(0), windowsizex(0), 
		windowsizey(0), windowtype(0), chatusersplit(0), stealth(false) { };
	FavoriteHubEntry(const HubEntry& rhs) throw() : name(rhs.getName()), server(rhs.getServer()), 
		description(rhs.getDescription()), connect(false), windowposx(0), windowposy(0), windowsizex(0), 
		windowsizey(0), windowtype(0), chatusersplit(0), stealth(false) { };
	FavoriteHubEntry(const FavoriteHubEntry& rhs) throw() : userdescription(rhs.userdescription), name(rhs.getName()), 
		server(rhs.getServer()), description(rhs.getDescription()), password(rhs.getPassword()), connect(rhs.getConnect()), 
		nick(rhs.nick), windowposx(rhs.windowposx), windowposy(rhs.windowposy), windowsizex(rhs.windowsizex), 
		windowsizey(rhs.windowsizey), windowtype(rhs.windowtype), chatusersplit(rhs.chatusersplit), stealth(rhs.stealth)
	// CDM EXTENSION BEGINS FAVS
	, rawOne(rhs.rawOne)
	, rawTwo(rhs.rawOne)
	, rawThree(rhs.rawThree)
	, rawFour(rhs.rawFour)
	, rawFive(rhs.rawFive)
	// CDM EXTENSION ENDS
		 { };
	~FavoriteHubEntry() throw() { };
	
	const string& getNick(bool useDefault = true) const { 
		return (!nick.empty() || !useDefault) ? nick : SETTING(NICK);
	}

	void setNick(const string& aNick) { nick = aNick; };

	GETSETREF(string, userdescription, UserDescription);
	GETSETREF(string, name, Name);
	GETSETREF(string, server, Server);
	GETSETREF(string, description, Description);
	GETSETREF(string, password, Password);
	GETSET(bool, connect, Connect);
	GETSET(int, windowposx, WindowPosX);
	GETSET(int, windowposy, WindowPosY);
	GETSET(int, windowsizex, WindowSizeX);
	GETSET(int, windowsizey, WindowSizeY);
	GETSET(int, windowtype, WindowType);
	GETSET(int, chatusersplit, ChatUserSplit);
	GETSET(bool, stealth, Stealth);
	// CDM EXTENSION BEGINS FAVS
	GETSETREF(string, rawOne, RawOne);
	GETSETREF(string, rawTwo, RawTwo);
	GETSETREF(string, rawThree, RawThree);
	GETSETREF(string, rawFour, RawFour);
	GETSETREF(string, rawFive, RawFive);
	// CDM EXTENSION BEGINS
private:
	string nick;
};

// iDC++
class RecentHubEntry {
public:
	typedef RecentHubEntry* Ptr;
	typedef vector<Ptr> List;
	typedef List::iterator Iter;

	~RecentHubEntry() throw() { }	
	
	GETSETREF(string, name, Name);
	GETSETREF(string, server, Server);
	GETSETREF(string, description, Description);
	GETSETREF(string, users, Users);
	GETSET(bool, connect, Connect);
};
// iDC++

class HubManagerListener {
public:
	typedef HubManagerListener* Ptr;
	typedef vector<Ptr> List;
	typedef List::iterator Iter;
	enum Types {
		DOWNLOAD_STARTING,
		DOWNLOAD_FAILED,
		DOWNLOAD_FINISHED,
// iDC++
		RECENT_ADDED,
		RECENT_REMOVED,
// iDC++
		FAVORITE_ADDED,
		FAVORITE_REMOVED,
		USER_ADDED,
		USER_REMOVED
	};

	virtual void onAction(Types, RecentHubEntry*) throw() { }; // iDC++
	virtual void onAction(Types, FavoriteHubEntry*) throw() { };
	virtual void onAction(Types, const string&) throw() { };
	virtual void onAction(Types, const User::Ptr&) throw() { };
	virtual void onAction(Types) throw() { };
};

class SimpleXML;

/**
 * Public hub list, favorites (hub&user). Assumed to be called only by UI thread.
 */
class HubManager : public Speaker<HubManagerListener>, private HttpConnectionListener, public Singleton<HubManager>,
	private SettingsManagerListener
{
public:
	
	void refresh();

	FavoriteHubEntry::List& getFavoriteHubs() { return favoriteHubs; };

// iDC++
	RecentHubEntry::List& getRecentHubs() { return recentHubs; };
// iDC++

	User::List& getFavoriteUsers() { return users; };
	
	void addFavoriteUser(User::Ptr& aUser) { 
		if(find(users.begin(), users.end(), aUser) == users.end()) {
			users.push_back(aUser);
			aUser->setFavoriteUser(new FavoriteUser());
			fire(HubManagerListener::USER_ADDED, aUser);
			save();
		}
	}

	void removeFavoriteUser(User::Ptr& aUser) {
		User::Iter i = find(users.begin(), users.end(), aUser);
		if(i != users.end()) {
			aUser->setFavoriteUser(NULL);
			fire(HubManagerListener::USER_REMOVED, aUser);
			users.erase(i);
			save();
		}
	}

	void addFavorite(const FavoriteHubEntry& aEntry) {
		FavoriteHubEntry* f;

		FavoriteHubEntry::Iter i = getFavoriteHub(aEntry.getServer());
		if(i != favoriteHubs.end()) {
			return;
		}
		f = new FavoriteHubEntry(aEntry);
		favoriteHubs.push_back(f);
		fire(HubManagerListener::FAVORITE_ADDED, f);
		save();
	}

	void removeFavorite(FavoriteHubEntry* entry) {
		FavoriteHubEntry::Iter i = find(favoriteHubs.begin(), favoriteHubs.end(), entry);
		if(i == favoriteHubs.end()) {
			return;
		}
		
		fire(HubManagerListener::FAVORITE_REMOVED, entry);
		favoriteHubs.erase(i);
		delete entry;
		save();
	}

	FavoriteHubEntry* getFavoriteHubEntry(const string& aServer) {
		for(FavoriteHubEntry::Iter i = favoriteHubs.begin(); i != favoriteHubs.end(); ++i) {
			FavoriteHubEntry* hub = *i;
			if(Util::stricmp(hub->getServer(), aServer) == 0) {
				return hub;
			}
		}
		return NULL;
	}
	
// iDC++
	void addRecent(const RecentHubEntry& aEntry) {
		RecentHubEntry* f;

		RecentHubEntry::Iter i = getRecentHub(aEntry.getServer());
		if(i != recentHubs.end()) {
			return;
		}
		f = new RecentHubEntry(aEntry);
		recentHubs.push_back(f);
		fire(HubManagerListener::RECENT_ADDED, f);
		recentsave();
	}

	void removeRecent(RecentHubEntry* entry) {
		RecentHubEntry::Iter i = find(recentHubs.begin(), recentHubs.end(), entry);
		if(i == recentHubs.end()) {
			return;
		}
		
		fire(HubManagerListener::RECENT_REMOVED, entry);
		recentHubs.erase(i);
		delete entry;
		recentsave();
	}

	void removeallRecent() {
		recentHubs.clear();
		recentsave();
	}
// iDC++
	
	HubEntry::List getPublicHubs() {
		Lock l(cs);
		return publicHubs;
	}

	// CDM EXTENSION BEGINS (profiles)
	ClientProfile::List& getClientProfiles() {
		Lock l(cs);
		return clientProfiles;
	}
	
	ClientProfile addClientProfile(
		const string& name, 
		const string& version, 
		const string& tag, 
		const string& extendedTag, 
		const string& lock, 
		const string& pk, 
		const string& supports, 
		const string& testSUR, 
		const string& userConCom, 
		const string& status,
		const string& cheatingdescription, 
		int rawToSend,
		int tagVersion, 
		int useExtraVersion, 
		int checkMismatch ) 
	{
		Lock l(cs);
		clientProfiles.push_back(
			ClientProfile(
			lastProfile++, 
			name, 
			version, 
			tag, 
			extendedTag, 
			lock, 
			pk, 
			supports, 
			testSUR, 
			userConCom, 
			status, 
			cheatingdescription,
			rawToSend, 
			tagVersion, 
			useExtraVersion, 
			checkMismatch
			)
		);
		return clientProfiles.back();
	}

	void addClientProfile( const StringList& sl ) {
		Lock l(cs);
		clientProfiles.push_back(
			ClientProfile(
			lastProfile++, 
			sl[0], 
			sl[1], 
			sl[2], 
			sl[3], 
			sl[4], 
			sl[5], 
			sl[6], 
			sl[7], 
			sl[8], 
			sl[9], 
			"", 
			0, 
			0, 
			0, 
			0
			)
		);
		sortPriorities();
		save();
	}

	int getProfileListSize() {
		return clientProfiles.size();
	}

	bool getClientProfile(int id, ClientProfile& cp) {
		Lock l(cs);
		for(ClientProfile::Iter i = clientProfiles.begin(); i != clientProfiles.end(); ++i) {
			if(i->getId() == id) {
				cp = *i;
				return true;
			}
		}
		return false;
	}

	bool getClientProfileByPosition(int pos, ClientProfile& cp) {
		Lock l(cs);
		for(ClientProfile::Iter i = clientProfiles.begin(); i != clientProfiles.end(); ++i) {
			if(i->getPriority() == pos) {
				cp = *i;
				return true;
			}
		}
		return false;
	}

	void removeClientProfile(int id) {
		Lock l(cs);
		for(ClientProfile::Iter i = clientProfiles.begin(); i != clientProfiles.end(); ++i) {
			if(i->getId() == id) {
				clientProfiles.erase(i);
				break;
			}
		}
	}

	void updateClientProfile(const ClientProfile& cp) {
		Lock l(cs);
		for(ClientProfile::Iter i = clientProfiles.begin(); i != clientProfiles.end(); ++i) {
			if(i->getId() == cp.getId()) {
				*i = cp;
				break;
			}
		}
	}

	bool moveClientProfile(int id, int pos) {
		dcassert(pos == -1 || pos == 1);
		Lock l(cs);
		for(ClientProfile::Iter i = clientProfiles.begin(); i != clientProfiles.end(); ++i) {
			if(i->getId() == id) {
				swap(*i, *(i + pos));
				return true;
			}
		}
		return false;
	}

	void sortPriorities() {
		Lock l(cs);
		int j = 1;
		for(ClientProfile::Iter i = clientProfiles.begin(); i != clientProfiles.end(); ++i) {
			i->setPriority(j++);
		}
	}
	// CDM EXTENSION ENDS

	UserCommand addUserCommand(int type, int ctx, int flags, const string& name, const string& command, const string& hub) {
		// No dupes, add it...
		Lock l(cs);
		userCommands.push_back(UserCommand(lastId++, type, ctx, flags, name, command, hub));
		UserCommand& uc = userCommands.back();
		if(!uc.isSet(UserCommand::FLAG_NOSAVE)) 
			save();
		return userCommands.back();
	}

	bool getUserCommand(int id, UserCommand& uc) {
		Lock l(cs);
		for(UserCommand::Iter i = userCommands.begin(); i != userCommands.end(); ++i) {
			if(i->getId() == id) {
				uc = *i;
				return true;
			}
		}
		return false;
	}

	bool moveUserCommand(int id, int pos) {
		dcassert(pos == -1 || pos == 1);
		Lock l(cs);
		for(UserCommand::Iter i = userCommands.begin(); i != userCommands.end(); ++i) {
			if(i->getId() == id) {
				swap(*i, *(i + pos));
				return true;
			}
		}
		return false;
	}

	void updateUserCommand(const UserCommand& uc) {
		bool nosave = true;
		Lock l(cs);
		for(UserCommand::Iter i = userCommands.begin(); i != userCommands.end(); ++i) {
			if(i->getId() == uc.getId()) {
				*i = uc;
				nosave = uc.isSet(UserCommand::FLAG_NOSAVE);
				break;
			}
		}
		if(!nosave)
			save();
	}

	void removeUserCommnad(int id) {
		bool nosave = true;
		Lock l(cs);
		for(UserCommand::Iter i = userCommands.begin(); i != userCommands.end(); ++i) {
			if(i->getId() == id) {
				nosave = i->isSet(UserCommand::FLAG_NOSAVE);
				userCommands.erase(i);
				break;
			}
	}
		if(!nosave)
			save();
	}
	void removeUserCommand(const string& srv) {
		Lock l(cs);
		for(UserCommand::Iter i = userCommands.begin(); i != userCommands.end(); ) {
			if((i->getHub() == srv) && i->isSet(UserCommand::FLAG_NOSAVE)) {
				i = userCommands.erase(i);
			} else {
				++i;
			}
		}
	}

	UserCommand::List getUserCommands() { Lock l(cs); return userCommands; };
	UserCommand::List getUserCommands(int ctx, const string& hub, bool op);

	bool isDownloading() { return running; };

// iDC++	
	void recentsave();
// iDC++
	void load();
	void save();
private:
	
	enum {
		TYPE_NORMAL,
		TYPE_BZIP2,
		// XML addition
		TYPE_XML,
		TYPE_XMLBZIP2
		//
	} listType;

	HubEntry::List publicHubs;
	FavoriteHubEntry::List favoriteHubs;
	RecentHubEntry::List recentHubs; // iDC++
	UserCommand::List userCommands;
	User::List users;
	// CDM EXTENSION BEGINS (profiles)
	ClientProfile::List clientProfiles;
	int lastProfile;
	// CDM EXTENSION ENDS

	CriticalSection cs;
	bool running;
	HttpConnection* c;
	int lastServer;
	int lastId;

	/** Used during loading to prevent saving. */
	bool dontSave;

	friend class Singleton<HubManager>;
	
	HubManager() : running(false), c(NULL), lastServer(0), lastId(0), dontSave(false), lastProfile(0) {
		SettingsManager::getInstance()->addListener(this);
	}

	virtual ~HubManager() {
		SettingsManager::getInstance()->addListener(this);
		if(c) {
			c->removeListener(this);
			delete c;
			c = NULL;
		}
		
		for_each(favoriteHubs.begin(), favoriteHubs.end(), DeleteFunction<FavoriteHubEntry*>());
		for_each(recentHubs.begin(), recentHubs.end(), DeleteFunction<RecentHubEntry*>());

	}
	
	string downloadBuf;
	
// iDC++
	RecentHubEntry::Iter getRecentHub(const string& aServer) {
		for(RecentHubEntry::Iter i = recentHubs.begin(); i != recentHubs.end(); ++i) {
			if((*i)->getServer() == aServer) {
				return i;
			}
		}
		return recentHubs.end();
	}
// iDC++

	FavoriteHubEntry::Iter getFavoriteHub(const string& aServer) {
		for(FavoriteHubEntry::Iter i = favoriteHubs.begin(); i != favoriteHubs.end(); ++i) {
			if(Util::stricmp((*i)->getServer(), aServer) == 0) {
				return i;
			}
		}
		return favoriteHubs.end();
	}

	// HttpConnectionListener
	virtual void onAction(HttpConnectionListener::Types type, HttpConnection* /*conn*/, const u_int8_t* buf, int len) throw();
	virtual void onAction(HttpConnectionListener::Types type, HttpConnection* /*conn*/, const string& aLine) throw();
	virtual void onAction(HttpConnectionListener::Types type, HttpConnection* /*conn*/) throw();
	
 	void onHttpFinished() throw();

	// SettingsManagerListener
	virtual void onAction(SettingsManagerListener::Types type, SimpleXML*) throw();

	void load(SimpleXML* aXml);
	
// iDC++
	void recentload(SimpleXML* aXml);
	void recentload();
// iDC++
};

#endif // !defined(AFX_HUBMANAGER_H__75858D5D_F12F_40D0_B127_5DDED226C098__INCLUDED_)

/**
 * @file
 * $Id: HubManager.h,v 1.48 2003/11/27 10:33:15 arnetheduck Exp $
 */

