﻿-- Current Mod Title
GroupCalendar_cTitle = string.format(GroupCalendar_cTitle, gGroupCalendar_VersionString);

-- 
local	GroupCalendar_cSettingsFormat = 3;

-- Constant for hiding Passwords
local	GroupCalendar_cObfuscatedPassword = "******";

-- Used to define the Link Format for Items
GroupCalendar_cItemLinkFormat = "|(%x+)|Hitem:(-?%d+):(-?%d+):(-?%d+):(-?%d+):-?%d+:-?%d+:-?%d+:-?%d+|h%[([^%]]+)%]|h|r";

-- Settings for Mod
gGroupCalendar_Settings =
{
	Format = GroupCalendar_cSettingsFormat,
	Debug = false,
	ShowEventsInLocalTime = false,
	PlayerSettings = {},
	RealmSettings = {},
};

gGroupCalendar_PlayerSettings = nil;
gGroupCalendar_RealmSettings = nil;

gGroupCalendar_PlayerName = nil;
gGroupCalendar_PlayerGuild = nil;
gGroupCalendar_PlayerLevel = nil;
gGroupCalendar_PlayerFactionGroup = nil;
gGroupCalendar_PlayerGuildRank = nil;
gGroupCalendar_RealmName = GetRealmName();
gGroupCalendar_Initialized = false;
gGroupCalendar_NewEvents = {};

gGroupCalendar_ServerTimeZoneOffset = 0; -- Offset that, when added to the server time yields the local time

gGroupCalendar_ActiveDialog = nil;

-- Panel Setup --

gGroupCalendar_PANEL_NULL = 0;
gGroupCalendar_PANEL_CALENDAR = 1;
gGroupCalendar_PANEL_SETUP = 2;
gGroupCalendar_PANEL_ABOUT = 3;

gGroupCalendar_PanelFrames =
{
	"GroupCalendarCalendarFrame",
	"GroupCalendarSetupFrame",
	"GroupCalendarAboutFrame",
};

gGroupCalendar_CurrentPanel = gGroupCalendar_PANEL_CALENDAR;

-- Panel Details End --


function GroupCalendar_OnLoad()
	SlashCmdList["CALENDAR"] = GroupCalendar_ExecuteCommand;
	
	SLASH_CALENDAR1 = "/calendar";
	
	tinsert(UISpecialFrames, "GroupCalendarFrame");
	
	UIPanelWindows["GroupCalendarFrame"] = {area = "left", pushable = 5, whileDead = 1};
	
	gGroupCalendar_PlayerName = UnitName("player");
	gGroupCalendar_PlayerLevel = UnitLevel("player");
	
	-- Register events
	
	GroupCalendar_RegisterEvent(this, "VARIABLES_LOADED", GroupCalendar_VariablesLoaded);
	GroupCalendar_RegisterEvent(this, "PLAYER_ENTERING_WORLD", GroupCalendar_PlayerEnteringWorld);
	
	-- For updating auto-config settings and guild trust
	-- values
	
	GroupCalendar_RegisterEvent(this, "GUILD_ROSTER_UPDATE", GroupCalendar_GuildRosterUpdate);
	GroupCalendar_RegisterEvent(this, "PLAYER_GUILD_UPDATE", GroupCalendar_PlayerGuildUpdate);
	
	-- For updating the enabled events when the players
	-- level changes
	
	GroupCalendar_RegisterEvent(this, "PLAYER_LEVEL_UP", GroupCalendar_PlayerLevelUp);
	
	-- For monitoring the status of the chat channel
	
	GroupCalendar_RegisterEvent(this, "CHAT_MSG_WHISPER", GroupCalendar_ChatMsgWhisper);
	
	-- For suspending/resuming the chat channel during logout
	
	GroupCalendar_RegisterEvent(this, "PLAYER_CAMPING", CalendarNetwork_SuspendChannel);
	GroupCalendar_RegisterEvent(this, "PLAYER_QUITING", CalendarNetwork_SuspendChannel);
	GroupCalendar_RegisterEvent(this, "LOGOUT_CANCEL", CalendarNetwork_ResumeChannel);
	
	-- For monitoring tradeskill cooldowns
	
	GroupCalendar_RegisterEvent(this, "TRADE_SKILL_UPDATE", EventDatabase_UpdateCurrentTradeskillCooldown);
	GroupCalendar_RegisterEvent(this, "TRADE_SKILL_SHOW", EventDatabase_UpdateCurrentTradeskillCooldown);
	GroupCalendar_RegisterEvent(this, "BAG_UPDATE_COOLDOWN", GroupCalendarCoolDowns_CheckItemCooldowns);
	GroupCalendar_RegisterEvent(this, "BAG_UPDATE", GroupCalendarCoolDowns_CheckItemCooldowns);
	GroupCalendar_RegisterEvent(this, "UPDATE_INSTANCE_INFO", EventDatabase_ScheduleSavedInstanceEvents);
	
	-- For managing group invites
	
	GroupCalendar_RegisterEvent(this, "PARTY_MEMBERS_CHANGED", CalendarGroupInvites_PartyMembersChanged);
	GroupCalendar_RegisterEvent(this, "RAID_ROSTER_UPDATE", CalendarGroupInvites_PartyMembersChanged);
	GroupCalendar_RegisterEvent(this, "PARTY_LOOT_METHOD_CHANGED", CalendarGroupInvites_PartyLootMethodChanged);
	
	-- For dragging the window
	
	this:RegisterForDrag("LeftButton");
	
	-- Tabs
	
	PanelTemplates_SetNumTabs(this, table.getn(gGroupCalendar_PanelFrames));
	GroupCalendarFrame.selectedTab = gGroupCalendar_CurrentPanel;
	PanelTemplates_UpdateTabs(this);
	
	-- Done initializing
	
	if DEFAULT_CHAT_FRAME then
		DEFAULT_CHAT_FRAME:AddMessage(GroupCalendar_cLoadMessage, 0.8, 0.8, 0.2);
	end
end

function GroupCalendar_RegisterEvent(pFrame, pEvent, pHandler)
	if not pHandler then
		MCDebugLib:ErrorMessage("GroupCalendar: Attemping to install a nil handler for event "..pEvent);
		return;
	end
	
	if not pFrame.EventHandlers then
		pFrame.EventHandlers = {};
	end
	
	pFrame.EventHandlers[pEvent] = pHandler;
	pFrame:RegisterEvent(pEvent);
end

function GroupCalendar_UnregisterEvent(pFrame, pEvent)
	if pFrame.EventHandlers then
		pFrame.EventHandlers[pEvent] = nil;
	end
	
	pFrame:UnregisterEvent(pEvent);
end

function GroupCalendar_DispatchEvent(pFrame, pEvent)
	if not pFrame.EventHandlers then	
		return false;
	end
	
	local	vEventHandler = pFrame.EventHandlers[pEvent];
	
	if not vEventHandler then
		MCDebugLib:ErrorMessage("GroupCalendar: No event handler for "..pEvent);
		return false;
	end
	
	vEventHandler(pEvent);
	return true;
end

function GroupCalendar_VariablesLoaded()
	gGroupCalendar_MinimumEventDate = Calendar_GetCurrentLocalDate() - gGroupCalendar_MaximumEventAge;
	
	if gGroupCalendar_Settings.TwentyFourHourTime ~= nil then
		if gGroupCalendar_Settings.TwentyFourHourTime then
			TwentyFourHourTime = nil;
		else
			TwentyFourHourTime = 1;
		end
	end
	
	EventDatabase_SetUserName(gGroupCalendar_PlayerName);
	EventDatabase_PlayerLevelChanged(gGroupCalendar_PlayerLevel);
	CalendarNetwork_CheckPlayerGuild();
	
	EventDatabase_Initialize();
	
	CalendarNetwork_CalendarLoaded();
	
	Calendar_SetStartWeekOnMonday(gGroupCalendar_Settings.StartOnMonday);
	
	if gGroupCalendar_Settings.DisableClock then
		GroupCalendarTime_HideAnalogClock();
	else
		GroupCalendarTime_ShowAnalogClock();
	end
end

function GroupCalendar_Reset()
	CalendarNetwork_Reset();
	
	-- Wipe the database
	
	gGroupCalendar_Database =
	{
		Format = GroupCalendar_cDatabaseFormat,
		Databases = {},
	};
	
	-- Reinitialize
	
	GroupCalendar_VariablesLoaded();
end

function GroupCalendar_OnShow()
	PlaySound("igCharacterInfoOpen");
	
	CalendarDisplay_StopFlashingReminder();
	
	local	vYear, vMonth, vDay, vHour, vMinute = Calendar_GetCurrentYearMonthDayHourMinute();
	local	vMonthStartDate = Calendar_ConvertMDYToDate(vMonth, 1, vYear);
	
	-- Update the guild roster
	
	if IsInGuild() and GetNumGuildMembers() == 0 then
		GuildRoster();
	end
	
	Calendar_SetDisplayDate(vMonthStartDate);
	Calendar_SetActualDate(vMonthStartDate + vDay - 1);
	GroupCalendar_ShowPanel(gGroupCalendar_PANEL_CALENDAR); -- Always switch  back to the Calendar view when showing the window
	
	GroupCalendarUseServerTime:SetChecked(not gGroupCalendar_Settings.ShowEventsInLocalTime);
end

function GroupCalendar_OnHide()
	PlaySound("igCharacterInfoClose");
	CalendarEventEditor_DoneEditing();
	CalendarEventViewer_DoneViewing();
	CalendarEditor_Close();
	GroupCalendarDatabasesList_Close();
	
	gGroupCalendar_NewEvents = {};
end

function GroupCalendar_OnEvent(pEvent)
	local	vLatencyStartTime;
	
	if gGroupCalendar_Settings.DebugLatency then
		vLatencyStartTime = GetTime();
	end
	
	if GroupCalendar_DispatchEvent(this, pEvent) then
		
		-- Handled
		
	elseif pEvent == "GUILD_ROSTER_UPDATE" then
		-- Ignore the update if we're not initialized yet
		
		if not gGroupCalendar_Initialized then
			return;
		end
		
		CalendarNetwork_GuildRosterChanged();
		CalendarGroupInvites_GuildRosterChanged();
		
	elseif pEvent == "PLAYER_GUILD_UPDATE" then
		CalendarNetwork_CheckPlayerGuild();

		-- Ignore the update if we're not initialized yet
		
		if not gGroupCalendar_Initialized then
			return;
		end
		
		GroupCalendar_UpdateEnabledControls();
		
	elseif pEvent == "PLAYER_LEVEL_UP" then
		gGroupCalendar_PlayerLevel = tonumber(arg1);
		EventDatabase_PlayerLevelChanged(gGroupCalendar_PlayerLevel);
		GroupCalendar_MajorDatabaseChange(nil);
	end
	
	--
	
	if gGroupCalendar_Settings.DebugLatency then
		local	vElapsed = GetTime() - vLatencyStartTime;
		
		if vElapsed > 0.1 then
			MCDebugLib:DebugMessage("Event "..pEvent.." took "..vElapsed.."s to execute");
		end
	end
end

function GroupCalendar_PlayerEnteringWorld(pEvent)
	gGroupCalendar_PlayerLevel = UnitLevel("player");
	gGroupCalendar_PlayerFactionGroup = UnitFactionGroup("player");
	gGroupCalendar_PlayerSettings = GroupCalendar_GetPlayerSettings(gGroupCalendar_PlayerName, GetRealmName());
	gGroupCalendar_RealmSettings = GroupCalendar_GetRealmSettings(GetRealmName());
	EventDatabase_PlayerLevelChanged(gGroupCalendar_PlayerLevel);
	GroupCalendarTime_CalculateTimeZoneOffset();
	GroupCalendar_MajorDatabaseChange(nil);
	
	MCSchedulerLib:ScheduleUniqueTask(30, GroupCalendar_CalculateReminders);
end

function GroupCalendar_ChatMsgWhisper(pEvent)
	CalendarWhisperLog_AddWhisper(arg2, arg1);
end

function GroupCalendar_GuildRosterUpdate(pEvent)
	-- Ignore the update if we're not initialized yet

	if not gGroupCalendar_Initialized then
		return;
	end

	CalendarNetwork_GuildRosterChanged();
	CalendarGroupInvites_GuildRosterChanged();
end

function GroupCalendar_PlayerGuildUpdate(pEvent)
	CalendarNetwork_CheckPlayerGuild();

	-- Ignore the update if we're not initialized yet
	
	if not gGroupCalendar_Initialized then
		return;
	end
	
	GroupCalendar_UpdateEnabledControls();
end

function GroupCalendar_PlayerLevelUp(pEvent)
	gGroupCalendar_PlayerLevel = tonumber(arg1);
	EventDatabase_PlayerLevelChanged(gGroupCalendar_PlayerLevel);
	GroupCalendar_MajorDatabaseChange(nil);
end

function GroupCalendar_UpdateEnabledControls()
	if GroupCalendarFrame.selectedTab == gGroupCalendar_PANEL_CALENDAR then
		-- Update the calendar display
		
	elseif GroupCalendarFrame.selectedTab == gGroupCalendar_PANEL_SETUP then
		local	vAutoConfig, vGuildAdmin = GroupCalendar_GetUIConfigMode();
		local	vAllowManualChanges = not vAutoConfig or vGuildAdmin;
		
		Calendar_SetCheckButtonEnable(GroupCalendarUseGuildChannel, vAllowManualChanges);
		
		if GroupCalendarUseGuildChannel:GetChecked() then
			Calendar_SetDropDownEnable(GroupCalendarTrustMinRank, vAllowManualChanges);
			if not UIDropDownMenu_GetSelectedValue(GroupCalendarTrustMinRank) then
				CalendarDropDown_SetSelectedValue(GroupCalendarTrustMinRank, 1);
			end
		else
			Calendar_SetDropDownEnable(GroupCalendarTrustMinRank, false);
		end
		
		Calendar_SetCheckButtonEnable(GroupCalendarUseSharedChannel, vAllowManualChanges);
		
		Calendar_SetEditBoxEnable(GroupCalendarChannelName, vAllowManualChanges and not GroupCalendarUseGuildChannel:GetChecked());
		Calendar_SetEditBoxEnable(GroupCalendarChannelPassword, vAllowManualChanges and not GroupCalendarUseGuildChannel:GetChecked());
		
		GroupCalendar_UpdateChannelStatus();
		
	elseif GroupCalendarFrame.selectedTab == gGroupCalendar_PANEL_ABOUT then
		-- Update the about frame
		
	end
end

function GroupCalendar_SetUseGuildChannel(pUseGuildChannel)
	GroupCalendarUseGuildChannel:SetChecked(pUseGuildChannel);
	GroupCalendarUseSharedChannel:SetChecked(not pUseGuildChannel);
	
	GroupCalendar_UpdateEnabledControls();
end

function GroupCalendar_ConfigModeChanged()
	GroupCalendar_UpdateEnabledControls();
end

function GroupCalendar_GetUIConfigMode()
	local	vAdmin = UIDropDownMenu_GetSelectedValue(CalendarConfigModeMenuConfigMode) == "CONFIG_ADMIN";
	local	vAutoConfig = vAdmin or UIDropDownMenu_GetSelectedValue(CalendarConfigModeMenuConfigMode) == "CONFIG_AUTO";
	
	return vAutoConfig, vAdmin;
end

function GroupCalendar_GetUIChannelInfo()
	local	vChannelName, vChannelPassword, vMinTrustedRank;

	if GroupCalendarUseGuildChannel:GetChecked() then
		vChannelName = "#GUILD";
		vMinTrustedRank = UIDropDownMenu_GetSelectedValue(GroupCalendarTrustMinRank);
		
		if not vMinTrustedRank then
			vMinTrustedRank = 1;
		end
	else
		vChannelName = GroupCalendarChannelName:GetText();
		
		if vChannelName == "" then
			vChannelName = nil;
		end
		
		vChannelPassword = GroupCalendarChannelPassword:GetText();
		
		if vChannelPassword == "" then
			vChannelPassword = nil;
		end
	end
	
	return vChannelName, vChannelPassword, vMinTrustedRank;
end

function GroupCalendar_SavePanel(pIndex)
	if pIndex == 2 then
		local	vAutoConfig, vGuildAdmin = GroupCalendar_GetUIConfigMode();
		local	vOrigPassword = gGroupCalendar_PlayerSettings.Channel.Password;
		
		gGroupCalendar_PlayerSettings.Channel.AutoConfig = vAutoConfig;
		gGroupCalendar_PlayerSettings.Channel.GuildAdmin = vGuildAdmin;
		
		gGroupCalendar_PlayerSettings.Channel.Name = nil;
		gGroupCalendar_PlayerSettings.Channel.Password = nil;
		gGroupCalendar_PlayerSettings.Channel.MinTrustedRank = nil;
		
		if not vAutoConfig or vGuildAdmin then
			-- Manual config and guild admin modes fetch the settings
			
			local	vChannelName, vChannelPassword, vMinTrustedRank = GroupCalendar_GetUIChannelInfo()
			
			if vGuildAdmin then
				CalendarNetwork_SetAutoConfigData(vChannelName, vChannelPassword, vMinTrustedRank);
			else
				gGroupCalendar_PlayerSettings.Channel.Name = vChannelName;
				gGroupCalendar_PlayerSettings.Channel.Password = vChannelPassword;
				gGroupCalendar_PlayerSettings.Channel.MinTrustedRank = vMinTrustedRank;
				
				CalendarNetwork_SetChannel(vChannelName, vChannelPassword);
				CalendarTrust_SetMinTrustedRank(vMinTrustedRank);
			end
		else
			-- Auto-config just schedule an update
			
			CalendarNetwork_ScheduleAutoConfig(0.5);
		end
		
	elseif pIndex == 3 then
		-- About panel
		
	end
	
	GroupCalendar_UpdateEnabledControls();
end

function GroupCalendar_ChannelPanelHasChanges()
	-- No changes if the panel isn't shown
	
	if not GroupCalendarFrame:IsVisible()
	or GroupCalendarFrame.selectedTab ~= 2 then
		return false;
	end
	
	--
	
	local	vAutoConfig, vGuildAdmin = GroupCalendar_GetUIConfigMode();
	
	if (vAutoConfig ~= gGroupCalendar_PlayerSettings.Channel.AutoConfig)
	or (vGuildAdmin ~= gGroupCalendar_PlayerSettings.Channel.GuildAdmin) then
		return true;
	end
	
	if not vAutoConfig or vGuildAdmin then
		local	vChannelInfo = GroupCalendar_GetCurrentChannelInfo();
		local	vChannelName, vChannelPassword, vMinTrustedRank = GroupCalendar_GetUIChannelInfo()
		
		if vChannelInfo.Name ~= vChannelName
		or vChannelInfo.Password ~= vChannelPassword
		or vChannelInfo.MinTrustedRank ~= vMinTrustedRank then
			return true;
		end
	end
	
	return false;
end

function GroupCalendar_UpdateChannelInfo(pChannelInfo)
	if pChannelInfo.Name == "#GUILD" then
		CalendarDropDown_SetSelectedValue(GroupCalendarTrustMinRank, pChannelInfo.MinTrustedRank);
	else
		CalendarDropDown_SetSelectedValue(GroupCalendarTrustMinRank, nil);
	end
	
	if gGroupCalendar_PlayerSettings.Channel.AutoConfig
	and not gGroupCalendar_PlayerSettings.Channel.GuildAdmin then
		if pChannelInfo.Password then
			GroupCalendarChannelPassword:SetText(GroupCalendar_cObfuscatedPassword);
		else
			GroupCalendarChannelPassword:SetText("");
		end
	else
		if pChannelInfo.Password then
			GroupCalendarChannelPassword:SetText(pChannelInfo.Password);
		else
			GroupCalendarChannelPassword:SetText("");
		end
	end
	
	if pChannelInfo.Name ~= nil
	and pChannelInfo.Name ~= "#GUILD" then
		GroupCalendarChannelName:SetText(pChannelInfo.Name);
	else
		GroupCalendarChannelName:SetText("");
	end
	
	GroupCalendar_UpdateChannelStatus();
end

function GroupCalendar_GetCurrentChannelInfo()
	local	vChannelName, vChannelPassword, vMinTrustedRank;

	if gGroupCalendar_PlayerSettings.Channel.AutoConfig
	or gGroupCalendar_PlayerSettings.Channel.GuildAdmin then
		local	vAutoConfigData = CalendarNetwork_GetAutoConfigData();
		
		if vAutoConfigData then
			vChannelName = vAutoConfigData.ChannelName;
			vChannelPassword = vAutoConfigData.ChannelPassword;
			vMinTrustedRank = vAutoConfigData.MinTrustedRank;
		end
	else
		vChannelName = gGroupCalendar_PlayerSettings.Channel.Name;
		vChannelPassword = gGroupCalendar_PlayerSettings.Channel.Password;
		vMinTrustedRank = gGroupCalendar_PlayerSettings.Channel.MinTrustedRank;
	end
	
	return {Name = vChannelName, Password = vChannelPassword, MinTrustedRank = vMinTrustedRank};
end

function GroupCalendar_ShowPanel(pPanelIndex)
	if gGroupCalendar_CurrentPanel > 0
	and gGroupCalendar_CurrentPanel ~= pPanelIndex then
		GroupCalendar_HidePanel(gGroupCalendar_CurrentPanel);
	end
	
	-- NOTE: Don't check for redundant calls since this function
	-- will be called to reset the field values as well as to 
	-- actuall show the panel when it's hidden
	
	gGroupCalendar_CurrentPanel = pPanelIndex;
	
	-- Hide the event editor/viewer if the calendar panel is being hidden
	
	if pPanelIndex ~= gGroupCalendar_PANEL_CALENDAR then
		CalendarEventEditor_DoneEditing();
		CalendarEventViewer_DoneViewing();
		CalendarEditor_Close();
	end
	
	if pPanelIndex ~= gGroupCalendar_PANEL_ABOUT then
		GroupCalendarDatabasesList_Close();
	end

	-- Update the control values
	
	if pPanelIndex == gGroupCalendar_PANEL_CALENDAR then
		-- Calendar display
		
	elseif pPanelIndex == gGroupCalendar_PANEL_SETUP then
		-- Channel panel
		
		CalendarConfigModeMenuConfigMode.ChangedValueFunc = GroupCalendar_ConfigModeChanged;
		
		if gGroupCalendar_PlayerSettings.Channel.GuildAdmin then
			CalendarDropDown_SetSelectedValue(CalendarConfigModeMenuConfigMode, "CONFIG_ADMIN");
		elseif gGroupCalendar_PlayerSettings.Channel.AutoConfig then
			CalendarDropDown_SetSelectedValue(CalendarConfigModeMenuConfigMode, "CONFIG_AUTO");
		else
			CalendarDropDown_SetSelectedValue(CalendarConfigModeMenuConfigMode, "CONFIG_MANUAL");
		end
		
		--
		
		local	vChannelInfo = GroupCalendar_GetCurrentChannelInfo();
		
		CalendarPlayerList_SetItemFunction(CalendarExcludedPlayersList, GroupCalendar_GetIndexedExcludedPlayer);
		CalendarPlayerList_SetSelectionChangedFunction(CalendarExcludedPlayersList, GroupCalendar_ExcludedPlayerSelected);
		
		GroupCalendarUseGuildChannel:SetChecked(vChannelInfo.Name == "#GUILD");
		GroupCalendarUseSharedChannel:SetChecked(vChannelInfo.Name ~= "#GUILD");
		
		GroupCalendar_UpdateChannelInfo(vChannelInfo);
	elseif pPanelIndex == gGroupCalendar_PANEL_ABOUT then
		-- About panel
		
	end
	
	GroupCalendar_UpdateEnabledControls();
	
	getglobal(gGroupCalendar_PanelFrames[pPanelIndex]):Show();
	
	PanelTemplates_SetTab(GroupCalendarFrame, pPanelIndex);
end

function GroupCalendar_HidePanel(pFrameIndex)
	if gGroupCalendar_CurrentPanel ~= pFrameIndex then
		return;
	end
	
	GroupCalendar_SavePanel(pFrameIndex);
	
	getglobal(gGroupCalendar_PanelFrames[pFrameIndex]):Hide();
	gGroupCalendar_CurrentPanel = gGroupCalendar_PANEL_NULL;
end

function GroupCalendar_UpdateChannelStatus()
	local	vChannelStatus, vStatusMessage, vStartTime = CalendarNetwork_GetChannelStatus();
	local	vStatusText = GroupCalendar_cChannelStatus[vChannelStatus];
	
	if not vStatusText then
		MCDebugLib:ErrorMessage("GroupCalendar: Unknown status "..vChannelStatus);
	end
	
	if vChannelStatus == "Synching" then
		local	vSecondsRemaining = gCalendarNetwork_RequestDelay.OwnedNotices - (GetTime() - vStartTime);
		local	vMinutesRemaining = math.floor(vSecondsRemaining / 60);
		
		vSecondsRemaining = vSecondsRemaining - vMinutesRemaining * 60;
		
		vStatusMessage = string.format(GroupCalendar_cTimeRemainingFormat, vMinutesRemaining, vSecondsRemaining);
		
		MCSchedulerLib:ScheduleUniqueTask(0.5, GroupCalendar_UpdateChannelStatus);
	else
		MCSchedulerLib:UnscheduleTask(GroupCalendar_UpdateChannelStatus);
	end
	
	GroupCalendarChannelStatus:SetText(string.format(vStatusText.mText, vStatusMessage));
	GroupCalendarChannelStatus:SetTextColor(vStatusText.mColor.r, vStatusText.mColor.g, vStatusText.mColor.b);
	
	if GroupCalendar_ChannelPanelHasChanges() then
		GroupCalendarConnectChannelButton:SetText(GroupCalendar_cApplyChannelChanges);
		Calendar_SetButtonEnable(GroupCalendarConnectChannelButton, true);
	
	elseif vChannelStatus == "Connected"
	or vChannelStatus == "Synching" then
		Calendar_SetButtonEnable(GroupCalendarConnectChannelButton, true);
		GroupCalendarConnectChannelButton:SetText(GroupCalendar_cDisconnectChannel);
	
	elseif vChannelStatus == "Disconnected" or vChannelStatus == "Error" then
		Calendar_SetButtonEnable(GroupCalendarConnectChannelButton, true);
		GroupCalendarConnectChannelButton:SetText(GroupCalendar_cConnectChannel);
	
	else
		Calendar_SetButtonEnable(GroupCalendarConnectChannelButton, false);
	end
end

function GroupCalendar_FixPlayerName(pName)
	if pName == nil
	or pName == "" then
		return nil;
	end
	
	local	vFirstChar = string.sub(pName, 1, 1);
	
	if (vFirstChar >= "a" and vFirstChar <= "z")
	or (vFirstChar >= "A" and vFirstChar <= "Z") then
		return string.upper(vFirstChar)..string.lower(string.sub(pName, 2));
	else
		return pName;
	end
end

function GroupCalendar_AddExcludedPlayer(pPlayerName)
	local	vPlayerName = GroupCalendar_FixPlayerName(pPlayerName);

	if vPlayerName == nil then
		return;
	end
	
	gGroupCalendar_PlayerSettings.Security.Player[vPlayerName] = 2;
	GroupCalendar_UpdateTrustedPlayerList();
	CalendarTrust_TrustSettingsChanged();
end

function GroupCalendar_RemoveTrustedPlayer(pPlayerName)
	local	vPlayerName = GroupCalendar_FixPlayerName(pPlayerName);

	if vPlayerName == nil then
		return;
	end
	
	gGroupCalendar_PlayerSettings.Security.Player[vPlayerName] = nil;
	
	GroupCalendar_UpdateTrustedPlayerList();
	
	CalendarPlayerList_SelectIndexedPlayer(CalendarExcludedPlayersList, 0);
	
	CalendarTrust_TrustSettingsChanged();
end

function GroupCalendar_UpdateTrustedPlayerList()
	CalendarPlayerList_Update(CalendarExcludedPlayersList);
end

function GroupCalendar_GetIndexedTrustedPlayer(pIndex)
	if pIndex == 0 then
		return CalendarTrust_GetNumTrustedPlayers(1);
	end
	
	return
	{
		Text = CalendarTrust_GetIndexedTrustedPlayers(1, pIndex);
	};
end

function GroupCalendar_GetIndexedExcludedPlayer(pIndex)
	if pIndex == 0 then
		return CalendarTrust_GetNumTrustedPlayers(2);
	end
	
	return
	{
		Text = CalendarTrust_GetIndexedTrustedPlayers(2, pIndex);
	};
end

function GroupCalendar_TrustedPlayerSelected(pIndex)
	if pIndex == 0 then
		return;
	end
	
	CalendarPlayerList_SelectIndexedPlayer(CalendarExcludedPlayersList, 0);
	
	local	vName = CalendarTrust_GetIndexedTrustedPlayers(1, pIndex);
	
	if vName then
		CalendarTrustedPlayerName:SetText(vName);
		CalendarTrustedPlayerName:HighlightText();
		CalendarTrustedPlayerName:SetFocus();
	end
end

function GroupCalendar_ExcludedPlayerSelected(pIndex)
	if pIndex == 0 then
		return;
	end
	
	local	vName = CalendarTrust_GetIndexedTrustedPlayers(2, pIndex);
	
	if vName then
		CalendarTrustedPlayerName:SetText(vName);
		CalendarTrustedPlayerName:HighlightText();
		CalendarTrustedPlayerName:SetFocus();
	end
end

function GroupCalendar_SelectDateWithToggle(pDate)
	if CalendarEditor_IsOpen()
	and gCalendarEditor_SelectedDate == pDate then
		CalendarEditor_Close();
	else
		GroupCalendar_SelectDate(pDate);
	end
end

function GroupCalendar_SelectDate(pDate)
	Calendar_SetSelectedDate(pDate);
	
	local	vCompiledSchedule = EventDatabase_GetCompiledSchedule(pDate);
	
	CalendarEditor_ShowCompiledSchedule(pDate, vCompiledSchedule);
end

function GroupCalendar_EditorClosed()
	Calendar_ClearSelectedDate();
end

function GroupCalendar_EventChanged(pDatabase, pEvent, pChangedFields)
	GroupCalendar_ScheduleChanged(pDatabase, pEvent.mDate);
	CalendarEventEditor_EventChanged(pEvent);
end

function GroupCalendar_ScheduleChanged(pDatabase, pDate)
	GroupCalendar_ScheduleChanged2(pDatabase, pDate);
	
	if gGroupCalendar_Settings.ShowEventsInLocalTime then
		if gGroupCalendar_ServerTimeZoneOffset < 0 then
			GroupCalendar_ScheduleChanged2(pDatabase, pDate - 1);
		elseif gGroupCalendar_ServerTimeZoneOffset > 0 then
			GroupCalendar_ScheduleChanged2(pDatabase, pDate + 1);
		end
	end
end

function GroupCalendar_ScheduleChanged2(pDatabase, pDate)
	local	vSchedule = pDatabase.Events[pDate];
	
	CalendarEditor_ScheduleChanged(pDate, pSchedule);
	Calendar_ScheduleChanged(pDate, pSchedule);
	CalendarEventViewer_ScheduleChanged(pDate);
	
	local	vCurrentDate, vCurrentTime = Calendar_GetCurrentServerDateTime();
	
	if pDate == vCurrentDate or pDate == vCurrentDate + 1 or pDate == vCurrentDate - 1 then
		MCSchedulerLib:ScheduleUniqueTask(5, GroupCalendar_CalculateReminders);
	end
end

function GroupCalendar_AddedNewEvent(pDatabase, pEvent)
	gGroupCalendar_NewEvents[pEvent.mDate] = true;
	
	if GroupCalendarFrame:IsVisible() then
		Calendar_StartFlashingDateButton(pEvent.mDate);
	else
		CalendarDisplay_StartFlashingReminder();
	end
end

function GroupCalendar_MajorDatabaseChange(pDatabase)
	CalendarEditor_MajorDatabaseChange();
	CalendarEventViewer_MajorDatabaseChange();
	CalendarEventEditor_MajorDatabaseChange();
	Calendar_MajorDatabaseChange();
end

function GroupCalendar_StartMoving()
	if not gGroupCalendar_PlayerSettings.UI.LockWindow then
		GroupCalendarFrame:StartMoving();
	end
end

function GroupCalendar_GetPlayerSettings(pPlayerName, pRealmName)
	if gGroupCalendar_Settings.Format < 2 then
		-- First migrate all player settings to the PlayerSettings and RealmSettings arrays
		
		local	vPlayerSettings = {};
		local	vRealmSettings = {};
		
		for vRealmPlayer, vSettings in pairs(gGroupCalendar_Settings) do
			if type(vSettings) == "table" then
				if string.find(vRealmPlayer, "_") then
					vPlayerSettings[vRealmPlayer] = vSettings;
				else
					vRealmSettings[vRealmPlayer] = vSettings;
				end
				
				gGroupCalendar_Settings[vRealmPlayer] = nil;
			end
		end
		
		gGroupCalendar_Settings.PlayerSettings = vPlayerSettings;
		gGroupCalendar_Settings.RealmSettings = vRealmSettings;
		
		for vRealmPlayer, vSettings in pairs(gGroupCalendar_Settings.PlayerSettings) do
			if vSettings.Channel.AutoConfig then
				vSettings.Channel.GuildAdmin = false;
				vSettings.Channel.Name = nil;
				vSettings.Channel.Password = nil;
				vSettings.Channel.AutoConfigPlayer = nil;
			elseif vSettings.Channel.AutoConfigPlayer then
				vSettings.Channel.GuildAdmin = IsInGuild() and CanEditGuildInfo();
				vSettings.Channel.AutoConfig = IsInGuild() and not CanEditGuildInfo();

				if vSettings.Security.MinTrustedRank then
					vSettings.Channel.UseGuildChannel = true;
					vSettings.Channel.Name = nil;
					vSettings.Channel.Password = nil;
				else
					vSettings.Channel.UseGuildChannel = false;
				end
				
				vSettings.Channel.AutoConfigPlayer = nil;
			else
				vSettings.Channel.GuildAdmin = false;
				
				if vSettings.Security.MinTrustedRank then
					vSettings.Channel.UseGuildChannel = true;
				else
					vSettings.Channel.UseGuildChannel = false;
				end
			end
			
			vSettings.Security.TrustAnyone = nil;
			vSettings.Security.TrustGuildies = nil;
		end -- for
		
		gGroupCalendar_Settings.Format = 2;
	end
	
	--
	
	local	vSettings = gGroupCalendar_Settings.PlayerSettings[pRealmName.."_"..pPlayerName];
	
	if vSettings == nil then
		vSettings =
		{
			Security =
			{
				MinTrustedRank = 1,
				Player = {},
			},
			
			Channel =
			{
				AutoConfig = IsInGuild() and not CanEditGuildInfo(),
				GuildAdmin = IsInGuild() and CanEditGuildInfo(),
				UseGuildChannel = IsInGuild(),
				Name = nil,
				Password = nil,
			},
			
			UI =
			{
				LockWindow = false,
			},
		};
		
		gGroupCalendar_Settings.PlayerSettings[pRealmName.."_"..pPlayerName] = vSettings;
	end
	
	return vSettings;
end

function GroupCalendar_GetRealmSettings(pRealmName)
	local	vSettings = gGroupCalendar_Settings.RealmSettings[pRealmName];
	
	if vSettings == nil then
		vSettings = {};
		gGroupCalendar_Settings.RealmSettings[pRealmName] = vSettings;
	end
	
	return vSettings;
end


function GroupCalendar_ChannelChanged()
	if GroupCalendarFrame:IsVisible()
	and gGroupCalendar_CurrentPanel == 2 then
		local	vAutoConfig, vGuildAdmin = GroupCalendar_GetUIConfigMode();
		
		if vAutoConfig or vGuildAdmin then
			GroupCalendar_ShowPanel(gGroupCalendar_CurrentPanel);
		end
		
		GroupCalendar_UpdateChannelStatus();
	end
end

function GroupCalendar_GetLocalizedStrings(pLocale)
	local	vStrings = {};
	
	-- Initialize the strings with copies form the enUS set
	
	for vIndex, vString in pairs(gCalendarLocalizedStrings.enUS) do
		vStrings[vIndex] = vString;
	end
	
	-- Select a set to use for overwriting
	
	local	vLocalizedStrings = gCalendarLocalizedStrings[pLocale];
	
	if not vLocalizedStrings then
		-- There's not a fit for the exact language/country specified, so just match the language
		
		local	vLanguageCode = string.sub(pLocale, 1, 2);
		
		vLocalizedStrings = GroupCalendar_SelectLanguage(vLanguageCode);
		
		if not vLocalizedStrings then
			return vStrings;
		end
	end
	
	-- Overwrise the english strings with translated strings
	
	for vIndex, vString in pairs(vLocalizedStrings) do
		vStrings[vIndex] = vString;
	end
	
	return vStrings;
end

function GroupCalendar_SelectLanguage(pLanguageCode)
	-- There's not a fit for the exact language/country specified, so just match the language

	local	vLanguageCode = string.sub(pLocale, 1, 2);

	for vLocale, vLocalizedStrings in pairs(gCalendarLocalizedStrings) do
		if pLanguageCode == string.sub(vLocale, 1, 2) then
			return vLocalizedStrings;
		end
	end
	
	-- No luck, hope they know english!
	
	return nil;
end

function GroupCalendar_ToggleChannelConnection()
	local	vChannelStatus = CalendarNetwork_GetChannelStatus();
	
	if vChannelStatus == "Initializing" then
		return;
	end
	
	if GroupCalendar_ChannelPanelHasChanges() then
		GroupCalendar_SavePanel(gGroupCalendar_CurrentPanel);
	
	elseif vChannelStatus == "Connected"
	or vChannelStatus == "Synching" then
		gGroupCalendar_Channel.Disconnected = true;
		CalendarNetwork_LeaveChannel();
	
	else
		GroupCalendar_SavePanel(gGroupCalendar_CurrentPanel);
		
		gGroupCalendar_Channel.Disconnected = false;
		
		CalendarNetwork_SetChannelStatus("Initializing");
		
		if gGroupCalendar_PlayerSettings.Channel.AutoConfig
		or gGroupCalendar_PlayerSettings.Channel.GuildAdmin then
			CalendarNetwork_ScheduleAutoConfig(0.5);
		elseif gGroupCalendar_PlayerSettings.Channel.Name then
			CalendarNetwork_SetChannel(
					gGroupCalendar_PlayerSettings.Channel.Name,
					gGroupCalendar_PlayerSettings.Channel.Password);
		else
			CalendarNetwork_SetChannelStatus("Disconnected");
		end
	end
end

function GroupCalendar_ToggleCalendarDisplay()
	if GroupCalendarFrame:IsVisible() then
		HideUIPanel(GroupCalendarFrame);
	else
		ShowUIPanel(GroupCalendarFrame);
	end
end



function GroupCalendar_BeginModalDialog(pDialogFrame)
	if gGroupCalendar_ActiveDialog then
		GroupCalendar_EndModalDialog(gGroupCalendar_ActiveDialog);
	end
	
	gGroupCalendar_ActiveDialog = pDialogFrame;
end

function GroupCalendar_EndModalDialog(pDialogFrame)
	if pDialogFrame ~= gGroupCalendar_ActiveDialog then
		return;
	end
	
	gGroupCalendar_ActiveDialog = nil;
	
	pDialogFrame:Hide();
end

function GroupCalendar_ExecuteCommand(pCommandString)
	local	vStartIndex, vEndIndex, vCommand, vParameter = string.find(pCommandString, "([^%s]+) ?(.*)");
	
	if not vCommand or vCommand == "" then
		ShowUIPanel(GroupCalendarFrame);
		return;
	end
	
	local	vCommandTable =
	{
		["help"] = {func = GroupCalendar_ShowCommandHelp},
		["versions"] = {func = GroupCalendar_DumpUserVersions},
		["reset"] = {func = GroupCalendar_AskReset},
		["kill"] = {func = GroupCalendar_AskKillUserDatabase},
		["mon"] = {func = GroupCalendarTime_SetMondayWeek},
		["sun"] = {func = GroupCalendarTime_SetSundayWeek},
		["12h"] = {func = GroupCalendarTime_Set12Hour},
		["24h"] = {func = GroupCalendarTime_Set24Hour},
		["clock"] = {func = GroupCalendar_SetClockOption},
		["reminder"] = {func = GroupCalendar_SetReminderOption},
	};
	
	local	vCommandInfo = vCommandTable[strlower(vCommand)];
	
	if not vCommandInfo then
		GroupCalendar_ShowCommandHelp();
		return;
	end
	
	vCommandInfo.func(vParameter);
end

function GroupCalendar_ShowCommandHelp()
	MCDebugLib:NoteMessage("GroupCalendar commands:");
	MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE.."/calendar help"..NORMAL_FONT_COLOR_CODE..": Shows this list of commands");
	MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE.."/calendar versions"..NORMAL_FONT_COLOR_CODE..": Displays the last known versions of GroupCalendar each user was running");
	MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE.."/calendar kill playerName"..NORMAL_FONT_COLOR_CODE..": Deletes all events for the player from the calendar (guild officers only)");
	MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE.."/calendar [sun|mon]"..NORMAL_FONT_COLOR_CODE..": Sets the starting day of the week");
	MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE.."/calendar [12h|24h]"..NORMAL_FONT_COLOR_CODE..": Sets WoW to use 12h or 24h times");
	MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE.."/calendar reset [all]"..NORMAL_FONT_COLOR_CODE..": Resets your calendar for all characters on the current realm.  Use the ALL switch to reset your calendars on all realms.");
	MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE.."/calendar clock [off|local|server|auto]"..NORMAL_FONT_COLOR_CODE..": Sets the display mode for the minimap clock");
	MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE.."/calendar reminder [on|off]"..NORMAL_FONT_COLOR_CODE..": Enables or disables event and cooldown reminders");
end



function GroupCalendar_GetUserVersionsList()
	local	vVersions = {};
	
	for vRealmUser, vDatabase in pairs(gGroupCalendar_Database.Databases) do
		if EventDatabase_DatabaseIsVisible(vDatabase)
		and not vDatabase.IsPlayerOwned then
			if vDatabase.AddonVersion then
				table.insert(vVersions, {UserName = vDatabase.UserName, Version = vDatabase.AddonVersion, Updated = vDatabase.AddonVersionUpdated});
			else
				-- Just ignore unknown versions
				-- table.insert(vVersions, {UserName = vDatabase.UserName, Version = "Unknown"});
			end
		end
	end
	
	table.sort(vVersions, GroupCalendar_CompareUserNameFields);
	
	return vVersions;
end

function GroupCalendar_DumpUserVersions()
	local	vVersions = GroupCalendar_GetUserVersionsList();
	
	for _, vVersion in pairs(vVersions) do
		MCDebugLib:NoteMessage(HIGHLIGHT_FONT_COLOR_CODE..vVersion.UserName..NORMAL_FONT_COLOR_CODE..": "..vVersion.Version);
	end
end

StaticPopupDialogs["CALENDAR_CONFIRM_RESET"] =
{
	text = TEXT(GroupCalendar_cConfirmResetMsg),
	button1 = TEXT(GroupCalendar_cReset),
	button2 = TEXT(CANCEL),
	OnAccept = function() GroupCalendar_Reset(); end,
	timeout = 0,
	whileDead = 1,
	hideOnEscape = 1,
};

StaticPopupDialogs["CALENDAR_CONFIRM_RESET_ALL"] =
{
	text = TEXT(GroupCalendar_cConfirmResetAllMsg),
	button1 = TEXT(GroupCalendar_cReset),
	button2 = TEXT(CANCEL),
	OnAccept = function() GroupCalendar_Reset(); end,
	timeout = 0,
	whileDead = 1,
	hideOnEscape = 1,
};

local	gGroupCalendar_ResetAll = false;

function GroupCalendar_AskReset(pParam)
	gGroupCalendar_ResetAll = string.upper(pParam) == "ALL";
	
	if gGroupCalendar_ResetAll then
		StaticPopup_Show("CALENDAR_CONFIRM_RESET_ALL");
	else
		StaticPopup_Show("CALENDAR_CONFIRM_RESET", gGroupCalendar_RealmName);
	end
end

function GroupCalendar_Reset()
	-- Remove all non-player databases and
	-- empty the player databases
	
	for vRealmUser, vDatabase in pairs(gGroupCalendar_Database.Databases) do
		if vDatabase.IsPlayerOwned then
			if gGroupCalendar_ResetAll
			or vDatabase.Realm == gGroupCalendar_RealmName then
				local	vDatabaseID = Calendar_GetCurrentDateTimeUT60();
				
				EventDatabase_PurgeDatabase(vDatabase, vDatabaseID);
				EventDatabase_PurgeRSVPs(vDatabase, vDatabaseID);
				CalendarNetwork_SendDatabaseNOU(vDatabase);
				CalendarNetwork_SendRSVPNOU(vDatabase);
			end
		else
			if gGroupCalendar_ResetAll
			or vDatabase.Realm == gGroupCalendar_RealmName then
				gGroupCalendar_Database.Databases[vRealmUser] = nil;
				GroupCalendar_MajorDatabaseChange(vDatabase);
			end
		end
	end
	
	CalendarNetwork_RequestExternalDatabaseUpdates();
end

StaticPopupDialogs["CALENDAR_CONFIRM_KILL"] =
{
	text = TEXT(GroupCalendar_cConfirmKillMsg),
	button1 = TEXT(GroupCalendar_cKill),
	button2 = TEXT(CANCEL),
	OnAccept = function() GroupCalendar_KillUserDatabase(); end,
	timeout = 0,
	whileDead = 1,
	hideOnEscape = 1,
};

local	gGroupCalendar_KillDatabaseUserName = nil;

function GroupCalendar_AskKillUserDatabase(pUserName)
	if not pUserName then
		MCDebugLib:ErrorMessage(GroupCalendar_cUserNameExpected);
		return;
	end
	
	if IsInGuild() and not CanEditOfficerNote() then
		MCDebugLib:ErrorMessage(GroupCalendar_cNotAnOfficerError);
		return;
	end
	
	local	vUserName = EventDatabase_FixUserName(pUserName);
	
	if CalendarNetwork_UserIsInSameGuild(vUserName) then
		MCDebugLib:ErrorMessage(format(GroupCalendar_cCantKillGuildieError, vUserName));
		return;
	end
	
	local	vDatabase = EventDatabase_GetDatabase(vUserName);
	
	if not vDatabase then
		MCDebugLib:ErrorMessage(format(GroupCalendar_cDatabaseNotFoundError, vUserName));
		return;
	end
	
	if IsInGuild() then
		gGroupCalendar_KillDatabaseUserName = vUserName;
		StaticPopup_Show("CALENDAR_CONFIRM_KILL", vUserName);
	else
		EventDatabase_DeleteDatabase(vUserName);
	end
end

function GroupCalendar_KillUserDatabase()
	-- Only allowed within a guild
	
	if not IsInGuild() then
		return;
	end
	
	if not CanEditOfficerNote() then
		return;
	end
	
	local	vDatabase = EventDatabase_GetDatabase(gGroupCalendar_KillDatabaseUserName);
	
	if not vDatabase then
		return;
	end
	
	-- Purge the events
	
	local	vDatabaseID = Calendar_GetCurrentDateTimeUT60();
	
	if vDatabaseID < vDatabase.Changes.ID  then
		vDatabaseID = vDatabase.Changes.ID + 1;
	end
	
	EventDatabase_PurgeDatabase(vDatabase, vDatabaseID);
	CalendarNetwork_SendDatabaseNOU(vDatabase);
	
	-- Purge the RSVPs
	
	if vDatabase.RSVPs then
		local	vDatabaseID = Calendar_GetCurrentDateTimeUT60();
		
		if vDatabaseID < vDatabase.RSVPs.ID  then
			vDatabaseID = vDatabase.RSVPs.ID + 1;
		end
		
		EventDatabase_PurgeRSVPs(vDatabase, vDatabaseID);
		CalendarNetwork_SendRSVPNOU(vDatabase);
	end
end

function GroupCalendar_CompareUserNameFields(pValue1, pValue2)
	return pValue1.UserName < pValue2.UserName;
end

function GroupCalendarDatabasesList_Open()
	local	vDesc =
	{
		Title = CalendarDatabases_cTitle,
		CloseFunc = GroupCalendarDatabasesList_Close,
		ButtonTitle = CalendarDatabases_cRefresh,
		ButtonDescription = CalendarDatabases_cRefreshDescription,
		ButtonFunc = GroupCalendarDatabasesList_Refresh,
		ListItems = GroupCalendarDatabasesList,
	};
	
	GroupCalendarDatabasesList.UpdateItems = GroupCalendarDatabasesList_UpdateItems;
	GroupCalendarDatabasesList.UpdateItem = GroupCalendarDatabasesList_UpdateItem;
	
	GroupCalendarDatabasesList.Versions = GroupCalendar_GetUserVersionsList();
	GroupCalendarSideList_Show(vDesc);
end

function GroupCalendarDatabasesList_Close()
	GroupCalendarSideList_Hide();
end

function GroupCalendarDatabasesList_Refresh()
	CalendarNetwork_RequestAllVersions(0);
end

function GroupCalendarDatabasesList_UpdateItems(pListItems)
	GroupCalendarSideList_SetNumItems(table.getn(GroupCalendarDatabasesList.Versions));
end

function GroupCalendarDatabasesList_UpdateItem(pListItems, pIndex, pItem, pItemName, pItemIndex)
	local	vLabelText = getglobal(pItemName.."Label");
	local	vValueText = getglobal(pItemName.."Value");
	local	vVersionInfo = pListItems.Versions[pIndex];
	
	vLabelText:SetText(vVersionInfo.UserName);
	vValueText:SetText(string.format(GroupCalendar_cShortVersionFormat, vVersionInfo.Version));
	
	pItem.VersionInfo = vVersionInfo;
	pItem.UpdateTooltip = GroupCalendarDatabasesList_UpdateTooltip;
end

function GroupCalendarDatabasesList_UpdateTooltip(pItem)
	GameTooltip:SetOwner(this, "ANCHOR_RIGHT");
	GameTooltip:AddLine(pItem.VersionInfo.UserName);
	
	GameTooltip:AddLine(string.format(GroupCalendar_cVersionFormat, pItem.VersionInfo.Version), 1, 1, 1, 1);
	
	if pItem.VersionInfo.Updated then
		local	vLocalDate, vLocalTime = Calendar_GetDateTimeFromTimeStamp(pItem.VersionInfo.Updated);
		
		GameTooltip:AddLine(string.format(GroupCalendar_cVersionUpdatedFormat, Calendar_GetLongDateString(vLocalDate), Calendar_GetShortTimeString(vLocalTime)), 1, 1, 1, 1);
	else
		GameTooltip:AddLine(GroupCalendar_cVersionUpdatedUnknown, 1, 1, 1, 1);
	end
	
	GameTooltip:Show();
end

function GroupCalendar_VersionDataChanged()
	if GroupCalendarDatabasesList:IsShown() then
		GroupCalendarDatabasesList.Versions = GroupCalendar_GetUserVersionsList();
		GroupCalendarDatabasesList:UpdateItems();
	end
end

function GroupCalendar_ToggleVersionsFrame()
	if GroupCalendarDatabasesList:IsShown() then
		GroupCalendarDatabasesList_Close();
	else
		GroupCalendarDatabasesList_Open();
	end
end

local	GroupCalendar_cClockOptions =
{
	["off"] = {DisableClock = true},
	["local"] = {DisableClock = false, ClockMode = "local"},
	["server"] = {DisableClock = false, ClockMode = "server"},
	["auto"] = {DisableClock = false, ClockMode = "auto"},
};

function GroupCalendar_SetClockOption(pOption)
	local	vOption = GroupCalendar_cClockOptions[string.lower(pOption)];
	
	if not vOption then
		MCDebugLib:ErrorMessage(GroupCalendar_cUnknownClockOption);
	end
	
	MCDebugLib:ErrorMessage("BLAH");
	
	for vKey, vValue in pairs(vOption) do
		gGroupCalendar_Settings[vKey] = vValue;
	end
	
	if gGroupCalendar_Settings.DisableClock then
		GroupCalendarTime_HideAnalogClock();
	else
		GroupCalendarTime_ShowAnalogClock();
	end
end


