﻿-- File to handle all the Reminder work

local	gGroupCalendar_Reminders = nil;
local	GroupCalendar_cReminderIntervals = {0, 60, 300, 900, 1800, 3600};
local	GroupCalendar_cNumReminderIntervals = table.getn(GroupCalendar_cReminderIntervals);

function GroupCalendar_SetReminderOption(pOption)
	if string.lower(pOption) == "off" then
		gGroupCalendar_Settings.DisableReminders = true;
		MCDebugLib:ErrorMessage("GroupCalendar: Reminders disabled");
	elseif string.lower(pOption) == "on" then
		gGroupCalendar_Settings.DisableReminders = nil;
		MCDebugLib:ErrorMessage("GroupCalendar: Reminders enabled");
		GroupCalendar_CalculateReminders();
	else
		MCDebugLib:ErrorMessage("GroupCalendar: Unknown reminder option");
	end
end

function GroupCalendar_EventNeedsReminder(pEvent, pOwner, pCurrentDateTimeStamp)
	-- Don't remind for events they're not attending
	
	if EventDatabase_EventTypeUsesAttendance(pEvent.mType)
	and not EventDatabase_PlayerIsAttendingEvent(pOwner, pEvent) then
		return false;
	end
	
	-- Don't remind for events which don't have a start time (birthdays and vacations)
	
	if not EventDatabase_EventTypeUsesTime(pEvent.mType) then
		return false;
	end
	
	-- Don't remind if all reminders have been issued
	
	if pEvent.ReminderIndex == 0 then
		return false;
	end
	
	-- Don't remind if the event has passed
	
	if pEvent.mDuration
	and GroupCalendar_GetTimeToEvent(pEvent, pCurrentDateTimeStamp) + pEvent.mDuration * 60 < 0 then
		return false;
	end
	
	-- Don't remind for dungeon resets
	
	if EventDatabase_IsDungeonResetEventType(pEvent.mType) then
		return false;
	end
	
	return true;
end

function GroupCalendar_CalculateReminders()
	-- Gather up events
	
	local	vCurrentDate, vCurrentTime = Calendar_GetCurrentServerDateTime();
	local	vCurrentDateTimeStamp = vCurrentDate * 86400 + vCurrentTime * 60;
	
	gGroupCalendar_Reminders = EventDatabase_GetCompiledSchedule(vCurrentDate - 1, nil, true);
	gGroupCalendar_Reminders = EventDatabase_GetCompiledSchedule(vCurrentDate, gGroupCalendar_Reminders, true);
	gGroupCalendar_Reminders = EventDatabase_GetCompiledSchedule(vCurrentDate + 1, gGroupCalendar_Reminders, true);
	
	-- Strip out events which don't need reminders
	
	local	vLastIndex = table.getn(gGroupCalendar_Reminders);
	local	vIndex = 1;
	
	while vIndex <= vLastIndex do
		local	vCompiledEvent = gGroupCalendar_Reminders[vIndex];
		
		if not GroupCalendar_EventNeedsReminder(vCompiledEvent.mEvent, vCompiledEvent.mOwner, vCurrentDateTimeStamp) then
			table.remove(gGroupCalendar_Reminders, vIndex);
			vLastIndex = vLastIndex - 1;
		else
			if not vCompiledEvent.mEvent.ReminderIndex then
				vCompiledEvent.mEvent.ReminderIndex = GroupCalendar_cNumReminderIntervals;
			end
			
			vIndex = vIndex + 1;
		end
	end
	
	MCDebugLib:DumpArray("Reminders", gGroupCalendar_Reminders);
	
	-- Calculate the time to the first event
	
	GroupCalendar_DoReminders();
end

function GroupCalendar_GetCompiledEventReminderInterval(pCompiledEvent, pCurrentDateTimeStamp)
	local	vTimeRemaining = GroupCalendar_GetTimeToEvent(pCompiledEvent.mEvent, pCurrentDateTimeStamp);
	local	vReminderIntervalPassed = false;
	
	while (vTimeRemaining - GroupCalendar_cReminderIntervals[pCompiledEvent.mEvent.ReminderIndex]) <= 0 do
		pCompiledEvent.mEvent.ReminderIndex = pCompiledEvent.mEvent.ReminderIndex - 1;
		
		if pCompiledEvent.mEvent.ReminderIndex > 0 then
			vReminderIntervalPassed = true;
			vTimeRemaining = GroupCalendar_GetTimeToEvent(pCompiledEvent.mEvent, pCurrentDateTimeStamp);
		else
			return nil, vTimeRemaining, vReminderIntervalPassed;
		end
	end
	
	return vTimeRemaining - GroupCalendar_cReminderIntervals[pCompiledEvent.mEvent.ReminderIndex], vTimeRemaining, vReminderIntervalPassed;
end

function GroupCalendar_DoReminders()
	local	vCurrentDateTimeStamp = Calendar_GetCurrentServerDateTimeStamp();
	local	vMinTimeRemaining = nil;
	local	vIndex = 1;
	
	if not gGroupCalendar_Settings.DisableReminders then
		while vIndex <= #gGroupCalendar_Reminders do
			local	vCompiledEvent = gGroupCalendar_Reminders[vIndex];
			local	vReminderTimeRemaining, vTimeRemaining, vReminderIntervalPassed = GroupCalendar_GetCompiledEventReminderInterval(vCompiledEvent, vCurrentDateTimeStamp);
			
			if vIndex == 1 then
				if vTimeRemaining <= 3600 then -- Show the icon for one hour before the event
					CalendarDisplay_ShowReminderIcon(Calendar_GetEventTypeIconPath(vCompiledEvent.mEvent.mType));
				else
					CalendarDisplay_HideReminderIcon();
				end
			end
			
			if vReminderIntervalPassed then
				local	vReminderTime = GroupCalendar_cReminderIntervals[vCompiledEvent.mEvent.ReminderIndex + 1];
				
				if vReminderTime == 0 then
					if vCompiledEvent.mEvent.mType == "Birth" then
						--[[ Disabling birthday reminders for now -- need an option or some way to control which birthdays a player cares about
						if vCompiledEvent.mEvent.mTitle then
							GroupCalendarMessages:AddMessage(EventDatabase_GetEventDisplayName(vCompiledEvent.mEvent));
						else
							GroupCalendarMessages:AddMessage(string.format(GroupCalendar_cHappyBirthdayFormat, vCompiledEvent.mOwner));
						end
						]]--
					elseif EventDatabase_IsResetEventType(vCompiledEvent.mEvent.mType) then
						GroupCalendarMessages:AddMessage(EventDatabase_GetEventDisplayName(vCompiledEvent.mEvent));
					elseif vTimeRemaining < -120 then
						GroupCalendarMessages:AddMessage(string.format(GroupCalendar_cAlreadyStartedFormat, EventDatabase_GetEventDisplayName(vCompiledEvent.mEvent)));
					else
						GroupCalendarMessages:AddMessage(string.format(GroupCalendar_cStartingNowFormat, EventDatabase_GetEventDisplayName(vCompiledEvent.mEvent)));
					end
				else
					local	vMinutesRemaining = math.floor(vTimeRemaining / 60 + 0.5);
					local	vFormat;
					
					if EventDatabase_IsResetEventType(vCompiledEvent.mEvent.mType) then
						if vMinutesRemaining == 1 then
							vFormat = GroupCalendar_cAvailableMinuteFormat;
						else
							vFormat = GroupCalendar_cAvailableMinutesFormat;
						end
					else
						if vMinutesRemaining == 1 then
							vFormat = GroupCalendar_cStartsMinuteFormat;
						else
							vFormat = GroupCalendar_cStartsMinutesFormat;
						end
					end
					
					GroupCalendarMessages:AddMessage(string.format(vFormat, EventDatabase_GetEventDisplayName(vCompiledEvent.mEvent), vMinutesRemaining));
				end
			end

			if vReminderTimeRemaining then
				if not vMinTimeRemaining or vReminderTimeRemaining < vMinTimeRemaining then
					vMinTimeRemaining = vReminderTimeRemaining;
				end
				
				if vCompiledEvent.mEvent.ReminderIndex == GroupCalendar_cNumReminderIntervals then
					break;
				end
				
				vIndex = vIndex + 1;
			else
				table.remove(gGroupCalendar_Reminders, vIndex);
			end
		end -- while
	end
	
	--
	
	MCSchedulerLib:UnscheduleTask(GroupCalendar_DoReminders);
	
	if vMinTimeRemaining then
		MCSchedulerLib:ScheduleTask(vMinTimeRemaining, GroupCalendar_DoReminders);
	else
		CalendarDisplay_HideReminderIcon();
	end
end

function GroupCalendar_DumpReminders()
	MCDebugLib:DumpArray("Reminders", gGroupCalendar_Reminders);
end

function GroupCalendar_GetTimeToEvent(pEvent, pCurrentDateTimeStamp)
	if not pOffset then
		pOffset = 0;
	end
	
	local	vEventDateTimeStamp = pEvent.mDate * 86400;
	
	if pEvent.mTime then
		vEventDateTimeStamp = vEventDateTimeStamp + pEvent.mTime * 60;
	end
	
	return vEventDateTimeStamp - pCurrentDateTimeStamp;
end
