local	gMCDataChannelLib_Version = 4;

if not MCDataChannelLib or MCDataChannelLib.Version < gMCDataChannelLib_Version then
	if not MCDataChannelLib then
		MCDataChannelLib =
		{
			ChannelInfo = {},
			EventIDs = {},
			Ready = false,
			
			SpecialChannelIDs =
			{
				["#GUILD"] = "GUILD",
				["#PARTY"] = "PARTY",
				["#RAID"] = "RAID",
				["#BATTLEGROUND"] = "BATTLEGROUND",
			},
			
			MaxSendBytesPerSecond = 450,
			NextSendTime = 0,
		};
	else
		MCDataChannelLib:UnregisterEvents();
	end
	
	function MCDataChannelLib:NewChannel(pPrefix, pMessageFunction, pClientRef)
		local	vChannelClient =
		{
			Prefix = pPrefix,
			MessageFunction = pMessageFunction,
			ClientRef = pClientRef,
			
			Open = false,
			Name = nil,
			Password = nil,
			
			Delete = self.Delete,
			
			OpenChannel = self.OpenChannel,
			CloseChannel = self.CloseChannel,
			
			SendMessage = self.SendMessage,
			SendClientMessage = self.SendClientMessage,
		};
		
		return vChannelClient;
	end
	
	function MCDataChannelLib:OpenChannelClient(pChannelClient, pChannelName, pPassword)
		-- Create the channel info if necessary
		
		local	vUpperChannelName = strupper(pChannelName);
		local	vChannelInfo = self.ChannelInfo[vUpperChannelName];
		local	vNewChannel = false;
		
		if not vChannelInfo then
			vChannelInfo =
			{
				Name = pChannelName,
				UpperName = vUpperChannelName,
				Password = pPassword,
				GotTooManyChannelsMessage = false,
				
				Clients = {},
				WildcardClients = {},
			};
			
			self.ChannelInfo[vUpperChannelName] = vChannelInfo;
			
			vChannelInfo.Permanent = self.SpecialChannelIDs[vUpperChannelName] ~= nil;
			
			if self.Ready then
				self:JoinChannel(vChannelInfo);
			else
				self:SetChannelStatus(vChannelInfo, "INIT", "Initializing");
			end
		end
		
		-- Add the client
		
		pChannelClient.Name = pChannelName;
		pChannelClient.Password = pPassword;
		
		if not pChannelClient.Prefix then
			table.insert(vChannelInfo.WildcardClients, pChannelClient);
		else
			vChannelInfo.Clients[pChannelClient.Prefix] = pChannelClient;
		end
		
		-- Notify the client of the current status
		
		pChannelClient.Status = vChannelInfo.Status;
		pChannelClient.StatusMessage = vChannelInfo.StatusMessage;
		pChannelClient:SendClientMessage("#STATUS", vChannelInfo.Status, vChannelInfo.StatusMessage);
		
		return true;
	end
	
	function MCDataChannelLib:CloseChannelClient(pChannelClient)
		local	vChannelInfo = self.ChannelInfo[strupper(pChannelClient.Name)];
		
		if not vChannelInfo then
			return false;
		end
		
		if not pChannelClient.Prefix then
			local	vFoundClient = false;
			
			for vIndex, vChannelClient in ipairs(vChannelInfo.WildcardClients) do
				if vChannelClient == pChannelClient then
					table.remove(vChannelInfo.WildcardClients, vIndex);
					vFoundClient = true;
					break;
				end
			end
			
			if not vFoundClient then
				return false;
			end
		else
			if vChannelInfo.Clients[pChannelClient.Prefix] ~= pChannelClient then
				return false;
			end
			
			vChannelInfo.Clients[pChannelClient.Prefix] = nil;
		end
		
		pChannelClient.Name = nil;
		pChannelClient.Password = nil;
		
		pChannelClient.Status = "DISCONNECTED";
		pChannelClient.StatusMessage = nil;
		pChannelClient:SendClientMessage("#STATUS", "DISCONNECTED");
		
		-- Just return if there are still more clients
		
		if #vChannelInfo.WildcardClients > 0 then
			return true;
		end
		
		for vClientPrefix, vChannelClient in pairs(vChannelInfo.Clients) do
			return true;
		end
		
		-- Otherwise shut down the channel
		
		self:LeaveChannel(vChannelInfo);
		self.ChannelInfo[vChannelInfo.UpperName] = nil;
		
		return true;
	end
	
	function MCDataChannelLib:JoinChannel(pChannelInfo)
		if pChannelInfo.Permanent then
			pChannelInfo.ID = self.SpecialChannelIDs[pChannelInfo.UpperName];
			
			self:SetChannelStatus(pChannelInfo, "CONNECTED");
		else
			pChannelInfo.ID = GetChannelName(pChannelInfo.Name);
			
			if not pChannelInfo.ID
			or pChannelInfo.ID == 0 then
				JoinChannelByName(pChannelInfo.Name, pChannelInfo.Password);
				pChannelInfo.ID = GetChannelName(pChannelInfo.Name);
				
				if not pChannelInfo.ID
				or pChannelInfo.ID == 0 then
					pChannelInfo.ID = nil;
					self:SetChannelStatus(pChannelInfo, "ERROR", "Joining channel failed");
					return false;
				end
				
				self:SetChannelStatus(pChannelInfo, "CONNECTING");
			else
				ChatFrame_RemoveChannel(DEFAULT_CHAT_FRAME, pChannelInfo.Name);
				self:SetChannelStatus(pChannelInfo, "CONNECTED");
			end
		end
	end
	
	function MCDataChannelLib:LeaveChannel(pChannelInfo)
		if not pChannelInfo.ID then
			return;
		end
		
		if not pChannelInfo.Permanent then
			LeaveChannelByName(pChannelInfo.Name);
		end
		
		pChannelInfo.ID = nil;
		
		self:SetChannelStatus(pChannelInfo, "DISCONNECTED");
	end
	
	function MCDataChannelLib:RegisterEvents()
		-- For suspending/resuming the chat channels during logout
		
		MCEventLib:RegisterEvent("PLAYER_CAMPING", MCDataChannelLib.SuspendChannels, MCDataChannelLib);
		MCEventLib:RegisterEvent("PLAYER_QUITING", MCDataChannelLib.SuspendChannels, MCDataChannelLib);
		MCEventLib:RegisterEvent("LOGOUT_CANCEL", MCDataChannelLib.ResumeChannels, MCDataChannelLib);

		MCEventLib:RegisterEvent("CHAT_MSG_CHANNEL_NOTICE", MCDataChannelLib.ChatMsgChannelNotice, MCDataChannelLib);

		MCEventLib:RegisterEvent("CHAT_MSG_ADDON", MCDataChannelLib.ChatMsgAddon, MCDataChannelLib);
		MCEventLib:RegisterEvent("CHAT_MSG_CHANNEL", MCDataChannelLib.ChatMsgChannel, MCDataChannelLib);
		MCEventLib:RegisterEvent("CHAT_MSG_SYSTEM", MCDataChannelLib.ChatMsgSystem, MCDataChannelLib);
	end
	
	function MCDataChannelLib:UnregisterEvents()
		MCEventLib:UnregisterEvent("PLAYER_CAMPING", MCDataChannelLib.SuspendChannels, MCDataChannelLib);
		MCEventLib:UnregisterEvent("PLAYER_QUITING", MCDataChannelLib.SuspendChannels, MCDataChannelLib);
		MCEventLib:UnregisterEvent("LOGOUT_CANCEL", MCDataChannelLib.ResumeChannels, MCDataChannelLib);

		MCEventLib:UnregisterEvent("CHAT_MSG_CHANNEL_NOTICE", MCDataChannelLib.ChatMsgChannelNotice, MCDataChannelLib);

		MCEventLib:UnregisterEvent("CHAT_MSG_ADDON", MCDataChannelLib.ChatMsgAddon, MCDataChannelLib);
		MCEventLib:UnregisterEvent("CHAT_MSG_CHANNEL", MCDataChannelLib.ChatMsgChannel, MCDataChannelLib);
		MCEventLib:UnregisterEvent("CHAT_MSG_SYSTEM", MCDataChannelLib.ChatMsgSystem, MCDataChannelLib);
	end
	
	function MCDataChannelLib:SuspendChannels()
		for _, vChannelInfo in pairs(self.ChannelInfo) do
			if not vChannelInfo.Suspended
			and vChannelInfo.Open then
				vChannelInfo.Suspended = true;
				self:LeaveChannel(vChannelInfo);
			end
		end -- for
	end
	
	function MCDataChannelLib:ResumeChannels()
		for _, vChannelInfo in pairs(self.ChannelInfo) do
			if vChannelInfo.Suspended then
				vChannelInfo.Suspended = false;
				self:JoinChannel(vChannelInfo);
			end
		end -- for
	end
	
	function MCDataChannelLib:SetChannelStatus(pChannelInfo, pStatus, pMessage)
		pChannelInfo.Status = pStatus;
		pChannelInfo.StatusMessage = pMessage;
		
		for vClientPrefix, vChannelClient in pairs(pChannelInfo.Clients) do
			vChannelClient.Status = pStatus;
			vChannelClient.StatusMessage = pMessage;
			vChannelClient:SendClientMessage("#STATUS", pStatus, pMessage);
		end

		for _, vChannelClient in pairs(pChannelInfo.WildcardClients) do
			vChannelClient.Status = pStatus;
			vChannelClient.StatusMessage = pMessage;
			vChannelClient:SendClientMessage("#STATUS", pStatus, pMessage);
		end
	end
	
	function MCDataChannelLib:SendClientMessage(pSender, pMessageID, pMessage)
		if self.ClientRef then
			self.MessageFunction(self.ClientRef, pSender, pMessageID, pMessage);
		else
			self.MessageFunction(pSender, pMessageID, pMessage);
		end
	end
	
	MCDataChannelLib.cSpecialChatChars = "`sS";

	MCDataChannelLib.cSpecialChatCharMap =
	{
		a = "`",
		l = "s",
		u = "S"
	};

	function MCDataChannelLib.EscapeChatString(pString)
		return string.gsub(
						pString,
						"(["..MCDataChannelLib.cSpecialChatChars.."])",
						function (pField)
							for vName, vChar in pairs(MCDataChannelLib.cSpecialChatCharMap) do
								if vChar == pField then
									return "`"..vName;
								end
							end
							
							return "";
						end);
	end

	function MCDataChannelLib.UnescapeChatString(pString)
		return string.gsub(
						pString,
						"`(.)", 
						function (pField)
							local	vChar = MCDataChannelLib.cSpecialChatCharMap[pField];
							
							if vChar ~= nil then
								return vChar;
							else
								return pField;
							end
						end);
	end
	
	MCDataChannelLib.cSpecialChars = ",/:;&|\n";

	MCDataChannelLib.cSpecialCharMap =
	{
		c = ",",
		s = "/",
		cn = ":",
		sc = ";",
		a = "&",
		b = "|",
		n = "\n",
	};

	function MCDataChannelLib.EscapeString(pString)
		return string.gsub(
						pString,
						"(["..MCDataChannelLib.cSpecialChars.."])",
						function (pField)
							for vName, vChar in pairs(MCDataChannelLib.cSpecialCharMap) do
								if vChar == pField then
									return "&"..vName..";";
								end
							end
							
							return "";
						end);
	end

	function MCDataChannelLib.UnescapeString(pString)
		return string.gsub(
						pString,
						"&(%w+);", 
						function (pField)
							local	vChar = MCDataChannelLib.cSpecialCharMap[pField];
							
							if vChar ~= nil then
								return vChar;
							else
								return pField;
							end
						end);
	end

	function MCDataChannelLib:SendChannelMessage(pChannelClient, pMessage)
		local	vChannelInfo = self.ChannelInfo[pChannelClient.UpperName];
		
		if not vChannelInfo
		or not vChannelInfo.ID then
			return;
		end
		
		if vChannelInfo.Permanent then
			-- MCDebugLib:TestMessage("To ["..vChannelInfo.Name.."/"..pChannelClient.Prefix.."]:"..pMessage);
			SendAddonMessage(pChannelClient.Prefix, pMessage, vChannelInfo.ID);
			return;
		end
		
		--
		
		local	vSavedAutoClearAFK = GetCVar("autoClearAFK");
		SetCVar("autoClearAFK", 0);
		
		if pChannelClient.Prefix then
			local	vMessage = pChannelClient.Prefix..":"..pMessage;
			SendChatMessage(self.EscapeChatString(vMessage), "CHANNEL", nil, vChannelInfo.ID);
			-- MCDebugLib:TestMessage("To ["..vChannelInfo.Name.."]:"..vMessage);
		else
			SendChatMessage(pMessage, "CHANNEL", nil, vChannelInfo.ID);
			-- MCDebugLib:TestMessage("To ["..vChannelInfo.Name.."]:"..pMessage);
		end
		
		SetCVar("autoClearAFK", vSavedAutoClearAFK);
	end
	
	function MCDataChannelLib:LibraryReady()
		if self.Ready then
			return;
		end
		
		self.Ready = true;
		
		-- Open any waiting channels

		for _, vChannelInfo in pairs(self.ChannelInfo) do
			self:JoinChannel(vChannelInfo);
		end
	end
	
	function MCDataChannelLib:ChatMsgChannel(pEventID, pNoticeID)
		-- See if it's a channel we're interested in
		
		local	vChannelName = arg9;
		
		if not vChannelName then
			return;
		end
		
		local	vUpperChannelName = strupper(vChannelName);
		local	vChannelInfo = self.ChannelInfo[vUpperChannelName];
		
		if not vChannelInfo then
			return;
		end
		
		-- Decode the message
		
		local	vMessage;
		
		if strsub(arg1, -8) == " ...hic!" then
			vMessage = self.UnescapeChatString(strsub(arg1, 1, -9));
		else
			vMessage = self.UnescapeChatString(arg1);
		end
		
		local	vStartIndex, vEndIndex, vPrefix, vMessageData = string.find(vMessage, "(%w+):(.*)");
		
		if not vStartIndex then
			for _, vChannelClient in pairs(vChannelInfo.WildcardClients) do
				vChannelClient:SendClientMessage(arg2, "DATA", arg1);
			end
			
			return;
		end
		
		local	vChannelClient = vChannelInfo.Clients[vPrefix];
		
		if vChannelClient then
			--MCDebugLib:TestMessage("["..vChannelName.."/"..vPrefix.."]["..arg2.."]: "..vMessage);
			vChannelClient:SendClientMessage(arg2, "DATA", vMessageData);
		end
	end
	
	function MCDataChannelLib:ChatMsgAddon(pEventID, pNoticeID)
		local	vPrefix = arg1;
		local	vChannelName = arg3;
		local	vSender = arg4;
		local	vMessage = arg2;
		
		local	vChannelInfo = self.ChannelInfo["#"..vChannelName];
		
		-- MCDebugLib:TestMessage("["..vChannelName.."/"..vPrefix.."]["..vSender.."]: "..vMessage);
		
		if vChannelInfo then
			local	vChannelClient = vChannelInfo.Clients[vPrefix];
			
			if not vChannelClient then
				return;
			end
			
			MCDebugLib:TestMessage("["..vChannelName.."/"..vPrefix.."]["..vSender.."]: "..vMessage);
			vChannelClient:SendClientMessage(vSender, "DATA", vMessage);
		end
		
		if vChannelName == "PARTY" then
			vChannelInfo = self.ChannelInfo["#RAID"];
			
			if vChannelInfo then
				local	vChannelClient = vChannelInfo.Clients[vPrefix];
				
				if not vChannelClient then
					return;
				end
				
				MCDebugLib:TestMessage("["..vChannelName.."/"..vPrefix.."]["..vSender.."]: "..vMessage);
				vChannelClient:SendClientMessage(vSender, "DATA", vMessage);
			end
		end
	end
	
	function MCDataChannelLib:ChatMsgSystem(pEventID, pMessage)
		if pMessage == ERR_TOO_MANY_CHAT_CHANNELS then
			for _, vChannelInfo in pairs(self.ChannelInfo) do
				if vChannelInfo.Status == "CONNECTING"
				or vChannelInfo.Status == "ERROR" then
					vChannelInfo.GotTooManyChannelsMessage = true;
					
					if vChannelInfo.Status == "ERROR" then
						self:SetChannelStatus(vChannelInfo, "ERROR", "Can't join more channels");
					end
				end
			end -- for
		end -- if
	end
	
	function MCDataChannelLib:ChatMsgChannelNotice(pEventID, pNoticeID)
		local	vChannelName = arg4;
		local	vChannelID = arg8;
		local	vActualChannelName = arg9;

		-- Decode the channel name if it's in the nn. <channel name> format
		
		local	_, _, vFoundNumber, vFoundName = string.find(vChannelName, "(%d+)%. (.+)");
		
		if vFoundName then
			vChannelName = vFoundName;
		end
		
		-- Just leave if it's not a channel we're interested in
		
		local	vUpperChannelName = strupper(vChannelName);
		local	vChannelInfo = self.ChannelInfo[vUpperChannelName];
		
		--
		
		if pNoticeID == "YOU_JOINED" then
			-- Once channels start showing up shorten the initialization delay
			
			if not self.Ready then
				MCSchedulerLib:SetTaskDelay(1, self.LibraryReady, self);
			end
			
			if not vChannelInfo then
				return;
			end
			
			vChannelInfo.ID = GetChannelName(vChannelInfo.Name);
			
			if not vChannelInfo.ID
			or vChannelInfo.ID == 0 then
				vChannelInfo.Open = false;
				vChannelInfo.ID = nil;
				self:SetChannelStatus(vChannelInfo, "ERROR", "Internal Error (Channel ID not found)");
			else
				vChannelInfo.Open = true;
				
				ChatFrame_RemoveChannel(DEFAULT_CHAT_FRAME, vChannelInfo.Name);
				self:SetChannelStatus(vChannelInfo, "CONNECTED");
			end
			
		elseif pNoticeID == "YOU_LEFT" then
			if not vChannelInfo then
				return;
			end
			
			if vChannelInfo.Open then
				vChannelInfo.Open = false;
				self:SetChannelStatus(vChannelInfo, "DISCONNECTED");
			end
		
		elseif pNoticeID == "WRONG_PASSWORD" then
			if not vChannelInfo then
				return;
			end
			
			self:SetChannelStatus(vChannelInfo, "ERROR", "Wrong password");
		end
	end
	
	-- Channel methods
	
	function MCDataChannelLib:Delete()
		if self.Open then
			self:CloseChannel();
		end
	end

	function MCDataChannelLib:OpenChannel(pChannelName, pPassword)
		if self.Open then
			self:CloseChannel();
		end
		
		if MCDataChannelLib:OpenChannelClient(self, pChannelName, pPassword) then
			self.Name = pChannelName;
			self.UpperName = strupper(pChannelName);
			self.Open = true;
		end
	end

	function MCDataChannelLib:CloseChannel()
		if not self.Open then
			return;
		end
		
		MCDataChannelLib:CloseChannelClient(self);
		self.Open = false;
	end

	function MCDataChannelLib:SendMessage(pMessage)
		if not self.DisableSend then
			MCDataChannelLib:SendChannelMessage(self, pMessage);
		end
	end
	
	-- Utilities
	
	function MCDataChannelLib.ParseCommandString(pCommandString)
		-- Break the command into parts
		
		local	vCommand = {};
		
		for vOpcode, vOperands in string.gmatch(pCommandString, "(%w+):?([^/]*)") do
			local	vOperation = {};
			
			vOperation.opcode = vOpcode;
			vOperation.operandString = vOperands;
			vOperation.operands = MCDataChannelLib.ParseParameterString(vOperands);
			
			table.insert(vCommand, vOperation);
		end
		
		return vCommand;
	end

	function MCDataChannelLib.ParseParameterString(pParameterString)
		local	vParameters = {};
		local	vIndex = 0;
		local	vFound = true;
		local	vStartIndex = 1;
		
		while vFound do
			local	vEndIndex;
			
			vFound, vEndIndex, vParameter = string.find(pParameterString, "([^,]*),", vStartIndex);
			
			vIndex = vIndex + 1;
			
			if not vFound then
				vParameters[vIndex] = string.sub(pParameterString, vStartIndex);
				break;
			end
			
			vParameters[vIndex] = vParameter;
			vStartIndex = vEndIndex + 1;
		end
		
		return vParameters;
	end
	
	function MCDataChannelLib.SendAddonMessage(pPrefix, pMessage, pChannel)
		local	vNumBytes = string.len(pMessage);
		local	vTime = GetTime();
		
		MCDataChannelLib.NextSendTime = MCDataChannelLib.NextSendTime + vNumBytes / MCDataChannelLib.MaxSendBytesPerSecond;
		
		if MCDataChannelLib.NextSendTime < vTime then
			MCDataChannelLib.NextSendTime = vTime;
		elseif MCDataChannelLib.NextSendTime > vTime + 0.5 then
			MCDataChannelLib.NextSendTime = vTime + 0.5;
		end
	end
	
	if not MCDataChannelLib.DidHookSendAddonMessage then
		MCDataChannelLib.DidHookSendAddonMessage = true;
		hooksecurefunc("SendAddonMessage", MCDataChannelLib.SendAddonMessage);
	end
	
	function MCDataChannelLib.SendChatMessage(pMessage, pType, pLanguage, pChannel)
		local	vNumBytes = string.len(pMessage);
		local	vTime = GetTime();
		
		MCDataChannelLib.NextSendTime = MCDataChannelLib.NextSendTime + vNumBytes / MCDataChannelLib.MaxSendBytesPerSecond;
		
		if MCDataChannelLib.NextSendTime < vTime then
			MCDataChannelLib.NextSendTime = vTime;
		elseif MCDataChannelLib.NextSendTime > vTime + 0.5 then
			MCDataChannelLib.NextSendTime = vTime + 0.5;
		end
	end
	
	if not MCDataChannelLib.DidHookSendChatMessage then
		MCDataChannelLib.DidHookSendChatMessage = true;
		hooksecurefunc("SendChatMessage", MCDataChannelLib.SendChatMessage);
	end
	
	--
	
	MCDataChannelLib.Version = gMCDataChannelLib_Version;
	MCDataChannelLib:RegisterEvents();
	
	if not MCDataChannelLib.Ready then
		local	vID1, vName1 = GetChannelList();
		
		-- If there are already channels then just signal that we're ready
		
		if vID1 then
			MCDataChannelLib:LibraryReady();
		
		-- Otherwise schedule a task to signal later after the world channels
		-- are joined
		
		else
			MCSchedulerLib:ScheduleTask(120, MCDataChannelLib.LibraryReady, MCDataChannelLib);
		end
	end
end
