local	MCDateLib_cVersion = 1;

if not MCDateLib or MCDateLib.Version < MCDateLib_cVersion then
	if not MCDateLib then
		MCDateLib = {};
	end
	
	MCDateLib.Version = MCDateLib_cVersion;
	
	MCDateLib.cDaysInMonth = {31, 28, 31, 30,  31,  30,  31,  31,  30,  31,  30,  31};
	MCDateLib.cDaysToMonth = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365};
	
	MCDateLib.cMinutesPerDay = 1440;
	MCDateLib.cSecondsPerDay = MCDateLib.cMinutesPerDay * 60;
	
	function MCDateLib:GetCurrentYearMonthDayHourMinute()
		local	vDate = date("*t");
		
		return vDate.year, vDate.month, vDate.day, vDate.hour, vDate.min, vDate.sec;
	end

	function MCDateLib:GetCurrentLocalDateTimeStamp()
		local	vDate, vTime60 = self:GetCurrentLocalDateTime60();
		
		return vDate * 86400 + vTime60;
	end

	function MCDateLib:GetDateTimeFromTimeStamp(pTimeStamp)
		return math.floor(pTimeStamp / 86400), math.floor(math.fmod(pTimeStamp, 86400) / 60);
	end

	function MCDateLib:GetCurrentLocalDateTime60()
		local	vDate = date("*t");
		
		return self:ConvertMDYToDate(vDate.month, vDate.day, vDate.year), self:ConvertHMSToTime60(vDate.hour, vDate.min, vDate.sec);
	end

	function MCDateLib:GetCurrentLocalDateTime()
		local	vDate = date("*t");
		
		return self:ConvertMDYToDate(vDate.month, vDate.day, vDate.year), self:ConvertHMToTime(vDate.hour, vDate.min);
	end

	function MCDateLib:GetCurrentUTCDateTime()
		local	vDate = date("!*t");
		
		return self:ConvertMDYToDate(vDate.month, vDate.day, vDate.year), self:ConvertHMToTime(vDate.hour, vDate.min);
	end

	function MCDateLib:GetCurrentLocalDate()
		local	vDate = date("*t");
		
		return self:ConvertMDYToDate(vDate.month, vDate.day, vDate.year);
	end

	function MCDateLib:GetCurrentLocalTime()
		local	vDate = date("*t");
		
		return self:ConvertHMToTime(vDate.hour, vDate.min);
	end

	function MCDateLib:GetCurrentUTCDateTime()
		local	vDate = date("!*t");
		
		return self:ConvertMDYToDate(vDate.month, vDate.day, vDate.year), self:ConvertHMToTime(vDate.hour, vDate.min);
	end

	function MCDateLib:GetCurrentUTCTime()
		local	vDate = date("!*t");
		
		return self:ConvertHMToTime(vDate.hour, vDate.min);
	end

	function MCDateLib:GetCurrentServerTime()
		return self:ConvertHMToTime(GetGameTime());
	end

	function MCDateLib:GetCurrentServerDateTime()
		return self:GetServerDateTimeFromLocalDateTime(self:GetCurrentLocalDateTime());
	end

	function MCDateLib:GetCurrentDateTimeUT60()
		local	vDate = date("!*t");
		
		return self:ConvertMDYToDate(vDate.month, vDate.day, vDate.year) * self.cSecondsPerDay
			 + self:ConvertHMSToTime60(vDate.hour, vDate.min, vDate.sec);
	end

	function MCDateLib:GetShortTimeString(pTime)
		if pTime == nil then
			return nil;
		end
		
		if TwentyFourHourTime then
			local	vHour, vMinute = self:ConvertTimeToHM(pTime);
			
			return format(TEXT(TIME_TWENTYFOURHOURS), vHour, vMinute);
		else
			local	vHour, vMinute, vAMPM = self:ConvertTimeToHMAMPM(pTime);
			
			if vAMPM == 0 then
				return format(TEXT(TIME_TWELVEHOURAM), vHour, vMinute);
			else
				return format(TEXT(TIME_TWELVEHOURPM), vHour, vMinute);
			end
		end
	end

	function MCDateLib:ConvertTimeToHM(pTime)
		local	vMinute = math.fmod(pTime, 60);
		local	vHour = (pTime - vMinute) / 60;
		
		return vHour, vMinute;
	end

	function MCDateLib:ConvertHMToTime(pHour, pMinute)
		return pHour * 60 + pMinute;
	end

	function MCDateLib:ConvertHMSToTime60(pHour, pMinute, pSecond)
		return pHour * 3600 + pMinute * 60 + pSecond;
	end

	function MCDateLib:ConvertTimeToHMAMPM(pTime)
		local	vHour, vMinute = self:ConvertTimeToHM(pTime);
		local	vAMPM;
		
		if vHour < 12 then
			vAMPM = 0;
			
			if vHour == 0 then
				vHour = 12;
			end
		else
			vAMPM = 1;

			if vHour > 12 then
				vHour = vHour - 12;
			end
		end

		return vHour, vMinute, vAMPM;
	end

	function MCDateLib:ConvertHMAMPMToTime(pHour, pMinute, pAMPM)
		local		vHour;
		
		if pAMPM == 0 then
			vHour = pHour;
			if vHour == 12 then
				vHour = 0;
			end
		else
			vHour = pHour + 12;
			if vHour == 24 then
				vHour = 12;
			end
		end
		
		return self:ConvertHMToTime(vHour, pMinute);
	end

	function MCDateLib:GetLongDateString(pDate, pIncludeDayOfWeek)
		local	vFormat;
		
		if pIncludeDayOfWeek then
			vFormat = MCDateLib.cLongDateFormatWithDayOfWeek;
		else
			vFormat = MCDateLib.cLongDateFormat;
		end
		
		return self:GetFormattedDateString(pDate, vFormat);
	end

	function MCDateLib:GetShortDateString(pDate, pIncludeDayOfWeek)
		return self:GetFormattedDateString(pDate, MCDateLib.cShortDateFormat);
	end

	function MCDateLib:FormatNamed(pFormat, pFields)
		return string.gsub(
						pFormat,
						"%$(%w+)", 
						function (pField)
							return pFields[pField];
						end);
	end

	function MCDateLib:GetFormattedDateString(pDate, pFormat)
		local	vMonth, vDay, vYear = self:ConvertDateToMDY(pDate);
		
		local	vDate =
				{
					dow = MCDateLib.cDayOfWeekNames[self:GetDayOfWeekFromDate(pDate) + 1],
					month = MCDateLib.cMonthNames[vMonth],
					monthNum = vMonth,
					day = vDay,
					year = vYear,
				};
		
		return self:FormatNamed(pFormat, vDate);
	end

	function MCDateLib:CalculateTimeZoneOffset()
		local	vServerTime = self:ConvertHMToTime(GetGameTime());
		local	vLocalDate, vLocalTime = self:GetCurrentLocalDateTime();
		local	vUTCDate, vUTCTime = self:GetCurrentUTCDateTime();
		
		local	vLocalDateTime = vLocalDate * 1440 + vLocalTime;
		local	vUTCDateTime = vUTCDate * 1440 + vUTCTime;
		
		local	vLocalUTCDelta = self:RoundTimeOffsetToNearest30(vLocalDateTime - vUTCDateTime);
		local	vLocalServerDelta = self:RoundTimeOffsetToNearest30(vLocalTime - vServerTime);
		local	vServerUTCDelta = vLocalUTCDelta - vLocalServerDelta;
		
		if vServerUTCDelta < (-12 * 60) then
			vServerUTCDelta = vServerUTCDelta + (24 * 60);
		elseif vServerUTCDelta > (12 * 60) then
			vServerUTCDelta = vServerUTCDelta - (24 * 60);
		end
		
		return vLocalUTCDelta - vServerUTCDelta;
	end

	function MCDateLib:RoundTimeOffsetToNearest30(pOffset)
		local	vNegativeOffset;
		local	vOffset;
		
		if pOffset < 0 then
			vNegativeOffset = true;
			vOffset = -pOffset;
		else
			vNegativeOffset = false;
			vOffset = pOffset;
		end
		
		vOffset = vOffset - (math.fmod(vOffset + 15, 30) - 15);
		
		if vNegativeOffset then
			return -vOffset;
		else
			return vOffset;
		end
	end

	function MCDateLib:GetDaysInMonth(pMonth, pYear)
		if pMonth == 2 and self:IsLeapYear(pYear) then
			return self.cDaysInMonth[pMonth] + 1;
		else
			return self.cDaysInMonth[pMonth];
		end
	end

	function MCDateLib:GetDaysToMonth(pMonth, pYear)
		if pMonth > 2 and self:IsLeapYear(pYear) then
			return self.cDaysToMonth[pMonth] + 1;
		elseif pMonth == 2 then
			return self.cDaysToMonth[pMonth];
		else
			return 0;
		end
	end

	function MCDateLib:GetDaysInYear(pYear)
		if self:IsLeapYear(pYear) then
			return 366;
		else
			return 365;
		end
	end

	function MCDateLib:IsLeapYear(pYear)
		return (math.fmod(pYear, 400) == 0)
		   or ((math.fmod(pYear, 4) == 0) and (math.fmod(pYear, 100) ~= 0));
	end

	function MCDateLib:GetDaysToDate(pMonth, pDay, pYear)
		local	vDays;
		
		vDays = self.cDaysToMonth[pMonth] + pDay - 1;
		
		if self:IsLeapYear(pYear) and pMonth > 2 then
			vDays = vDays + 1;
		end
		
		return vDays;
	end

	function MCDateLib:ConvertMDYToDate(pMonth, pDay, pYear)
		local		vDays = 0;
		
		for vYear = 2000, pYear - 1 do
			vDays = vDays + self:GetDaysInYear(vYear);
		end
		
		return vDays + self:GetDaysToDate(pMonth, pDay, pYear);
	end

	function MCDateLib:ConvertDateToMDY(pDate)
		local		vDays = pDate;
		local		vYear = 2000;
		local		vDaysInYear = self:GetDaysInYear(vYear);
		
		while vDays >= vDaysInYear do
			vDays = vDays - vDaysInYear;

			vYear = vYear + 1;
			vDaysInYear = self:GetDaysInYear(vYear);
		end
		
		local		vIsLeapYear = self:IsLeapYear(vYear);
		
		for vMonth = 1, 12 do
			local vDaysInMonth = self.cDaysInMonth[vMonth];
			
			if vMonth == 2 and vIsLeapYear then
				vDaysInMonth = vDaysInMonth + 1;
			end
			
			if vDays < vDaysInMonth then
				return vMonth, vDays + 1, vYear;
			end
			
			vDays = vDays - vDaysInMonth;
		end
		
		return 0, 0, 0;
	end

	function MCDateLib:GetDayOfWeek(pMonth, pDay, pYear)
		local	vDayOfWeek = 6; -- January 1, 2000 is a Saturday
		
		for vYear = 2000, pYear - 1 do
			if self:IsLeapYear(vYear) then
				vDayOfWeek = vDayOfWeek + 2;
			else
				vDayOfWeek = vDayOfWeek + 1;
			end
		end
		
		vDayOfWeek = vDayOfWeek + self:GetDaysToDate(pMonth, pDay, pYear);
		
		return math.fmod(vDayOfWeek, 7);
	end

	function MCDateLib:GetDayOfWeekFromDate(pDate)
		return math.fmod(pDate + 6, 7);  -- + 6 because January 1, 2000 is a Saturday
	end
end
