/*
 *  Copyright (C) 2006 Ludovic Jacomme (ludovic.jacomme@gmail.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL/SDL.h>
#include <SDL/SDL_mixer.h>
#include <SDL/SDL_ttf.h>

#include <pspctrl.h>
#include <pspkernel.h>
#include <pspdebug.h>
#include <pspiofilemgr.h>

#include "global.h"
#include "psp_sdl.h"
#include "psp_kbd.h"
#include "psp_menu.h"
#include "psp_fmgr.h"
#include "psp_menu_kbd.h"
#include "psp_battery.h"

extern SDL_Surface *back_surface;

# define MENU_SOUND        0
# define MENU_SKIP_FPS     1
# define MENU_RENDER       2
# define MENU_ANALOG       3
# define MENU_CLOCK        4
# define MENU_SCREENSHOT   5

# define MENU_CUR_SLOT     6
# define MENU_LOAD_SLOT    7
# define MENU_SAVE_SLOT    8
# define MENU_DEL_SLOT     9

# define MENU_LOAD_ROM    10
# define MENU_KEYBOARD    11

# define MENU_ORIGINAL    12
# define MENU_SAVE_CFG    13
# define MENU_RESET       14
# define MENU_BACK        15
# define MENU_EXIT        16

# define MAX_MENU_ITEM (MENU_EXIT + 1)

  static menu_item_t menu_list[] =
  { 
    "Sound enable        :",
    "Skip frame          :",
    "Render mode         :",
    "Swap Analog/Cursor  :",
    "Clock frequency     :",
    "Save Screenshot     :",
    "Change Slot",           
    "Load Slot",
    "Save Slot",
    "Delete Slot",
    "Load ROM",     
    "Keyboard", 
    "Atari800 menu",
    "Save Config",
    "Reset ATARI",
    "Back to ATARI",
    "Exit"
  };

  static int cur_menu_id = MENU_LOAD_ROM;
  static int cur_slot    = 0;

  static int atari_snd_enable       = 0;
  static int atari_render_mode      = 0;
  static int psp_reverse_analog   = 0;
  static int psp_cpu_clock        = 222;
  static int atari_skip_fps         = 0;

void 
psp_menu_display_save_name()
{
  char buffer[32];
  int Length;
  snprintf(buffer, 20, "Game: %s", ATARI.atari_save_name);
  Length = strlen(buffer);
  psp_sdl_fill_print(460 - (8*Length), 25, buffer, PSP_MENU_TEXT2_COLOR, PSP_MENU_BACKGROUND_COLOR);
}

void
psp_display_screen_menu_battery(void)
{
  char buffer[64];

  int Length;
  int color;

  snprintf(buffer, 50, " [%s] ", psp_get_battery_string());
  Length = strlen(buffer);

  if (psp_is_low_battery()) color = PSP_MENU_RED_COLOR;
  else                      color = PSP_MENU_GREEN_COLOR;

  psp_sdl_fill_print(240 - ((8*Length) / 2), 6, buffer, color, PSP_MENU_BACKGROUND_COLOR);
}

static void 
psp_display_screen_menu(void)
{
  char buffer[64];
  int menu_id = 0;
  int slot_id = 0;
  int color   = 0;
  int x       = 0;
  int y       = 0;
  int y_step  = 0;

  psp_sdl_blit_background();

  psp_sdl_draw_rectangle(10,10,459,249,PSP_MENU_BORDER_COLOR,0);
  psp_sdl_draw_rectangle(11,11,457,247,PSP_MENU_BORDER_COLOR,0);

  psp_sdl_fill_print( 30, 6, " Start+L+R: EXIT ", 
                     PSP_MENU_WARNING_COLOR,  PSP_MENU_BACKGROUND_COLOR);
  psp_display_screen_menu_battery();

  psp_sdl_fill_print( 370, 6, " R: RESET ", 
                     PSP_MENU_NOTE_COLOR,  PSP_MENU_BACKGROUND_COLOR);

  psp_sdl_fill_print(30, 254, " []: Cancel  O/X: Valid  SELECT: Back ", 
                     PSP_MENU_BORDER_COLOR, PSP_MENU_BACKGROUND_COLOR);
  psp_sdl_fill_print(370, 254, " By Zx-81 ",
                     PSP_MENU_AUTHOR_COLOR, PSP_MENU_BACKGROUND_COLOR);
  
  x      = 20;
  y      = 25;
  y_step = 10;
  
  for (menu_id = 0; menu_id < MAX_MENU_ITEM; menu_id++) {
    color = PSP_MENU_TEXT_COLOR;
    if (cur_menu_id == menu_id) color = PSP_MENU_SEL_COLOR;
    else 
    if (menu_id == MENU_EXIT) color = PSP_MENU_WARNING_COLOR;
    else 
    if (menu_id == MENU_ORIGINAL) color = PSP_MENU_NOTE_COLOR;

    psp_sdl_fill_print(x, y, menu_list[menu_id].title, color, PSP_MENU_BACKGROUND_COLOR);

    if (menu_id == MENU_SOUND) {
      if (atari_snd_enable) strcpy(buffer,"yes");
      else               strcpy(buffer,"no ");
      psp_sdl_fill_print(210, y, buffer, color, PSP_MENU_BACKGROUND_COLOR);
    } else
    if (menu_id == MENU_ANALOG) {
      if (psp_reverse_analog) strcpy(buffer,"yes");
      else                    strcpy(buffer,"no ");
      psp_sdl_fill_print(210, y, buffer, color, PSP_MENU_BACKGROUND_COLOR);
    } else
    if (menu_id == MENU_SKIP_FPS) {
      sprintf(buffer,"%d", atari_skip_fps);
      psp_sdl_fill_print(210, y, buffer, color, PSP_MENU_BACKGROUND_COLOR);
    } else
    if (menu_id == MENU_RENDER) {
      if (atari_render_mode == ATARI_RENDER_NORMAL    ) strcpy(buffer, "normal");
      else 
      if (atari_render_mode == ATARI_RENDER_FIT       ) strcpy(buffer, "fit");
      else
      if (atari_render_mode == ATARI_RENDER_FIT_WIDTH ) strcpy(buffer, "fit width");
      else                                              strcpy(buffer, "max");
      psp_sdl_fill_print(210, y, buffer, color, PSP_MENU_BACKGROUND_COLOR);
    } else
    if (menu_id == MENU_CLOCK) {
      sprintf(buffer,"%d", psp_cpu_clock);
      psp_sdl_fill_print(210, y, buffer, color, PSP_MENU_BACKGROUND_COLOR);
    } else
    if (menu_id == MENU_SCREENSHOT) {
      sprintf(buffer,"%d", ATARI.psp_screenshot_id);
      psp_sdl_fill_print(210, y, buffer, color, PSP_MENU_BACKGROUND_COLOR);
      y += y_step;
    } else
    if (menu_id == MENU_DEL_SLOT) {
      y += y_step;
    } else
    if (menu_id == MENU_KEYBOARD) {
      y += y_step;
    } else
    if (menu_id == MENU_BACK) {
      y += y_step;
    }

    y += y_step;
  }

  y_step = 10;
  y      = 95;

  for (slot_id = 0; slot_id < ATARI_MAX_SAVE_STATE; slot_id++) {
    if (slot_id == cur_slot) color = PSP_MENU_SEL2_COLOR;
    else                     color = PSP_MENU_TEXT_COLOR;

    if (ATARI.atari_save_used[slot_id]) sprintf(buffer, "%d - used" , slot_id + 1);
    else                            sprintf(buffer, "%d - empty", slot_id + 1);
    psp_sdl_fill_print(160, y, buffer, color, PSP_MENU_BACKGROUND_COLOR);

    y += y_step;
  }

  psp_menu_display_save_name();
}

static void
psp_main_menu_original(void)
{
  psp_kbd_run_command_ui();
}

static void
psp_main_menu_reset(void)
{
  /* Reset ! */
  psp_display_screen_menu();
  psp_sdl_fill_print(270,  80, "RESET ATARI !", 
                     PSP_MENU_WARNING_COLOR, PSP_MENU_BACKGROUND_COLOR);
  psp_sdl_flip();
  emulator_reset();
  sleep(1);
}

static void
psp_main_menu_clock(int step)
{
  if (step > 0) {
    if (psp_cpu_clock == 222) {
      psp_cpu_clock = 300;
    } else if (psp_cpu_clock == 300) {
      psp_cpu_clock = 333;
    } else {
      psp_cpu_clock = 222;
    }
  } else {
    if (psp_cpu_clock == 222) {
      psp_cpu_clock = 333;
    } else if (psp_cpu_clock == 333) {
      psp_cpu_clock = 300;
    } else {
      psp_cpu_clock = 222;
    }
  }
}

static void
psp_main_menu_skip_fps(int step)
{
  if (step > 0) {
    if (atari_skip_fps < 25) atari_skip_fps++;
  } else {
    if (atari_skip_fps >  0) atari_skip_fps--;
  }
}

static void
psp_main_menu_render(int step)
{
  if (step > 0) {
    if (atari_render_mode < ATARI_LAST_RENDER) atari_render_mode++;
    else                                   atari_render_mode = 0;
  } else {
    if (atari_render_mode > 0) atari_render_mode--;
    else                     atari_render_mode = ATARI_LAST_RENDER;
  }
}

static void
psp_main_menu_load(int format)
{
  int ret;

  ret = psp_fmgr_menu(format);
  if (ret ==  1) /* load OK */
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "File loaded !", 
                       PSP_MENU_NOTE_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
  else 
  if (ret == -1) /* Load Error */
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "Can't load file !", 
                       PSP_MENU_WARNING_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
}

static void
psp_main_menu_load_slot()
{
  int error;

  if (! ATARI.atari_save_used[cur_slot]) {

    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "Slot is empty !", 
                       PSP_MENU_NOTE_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);

    return;
  }

  error = atari_snapshot_load_slot(cur_slot);

  if (! error) /* save OK */
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "File loaded !", 
                       PSP_MENU_NOTE_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
  else 
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "Can't load file !", 
                       PSP_MENU_WARNING_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
}

static void
psp_main_menu_save_slot()
{
  int error;
  error = atari_snapshot_save_slot(cur_slot);

  if (! error) /* save OK */
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "File saved !", 
                       PSP_MENU_NOTE_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
  else 
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "Can't save file !", 
                       PSP_MENU_WARNING_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
}

static void
psp_main_menu_del_slot()
{
  int error;

  if (! ATARI.atari_save_used[cur_slot]) {

    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "Slot is empty !", 
                       PSP_MENU_NOTE_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);

    return;
  }

  error = atari_snapshot_del_slot(cur_slot);

  if (! error) /* save OK */
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "File deleted !", 
                       PSP_MENU_NOTE_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
  else 
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "Can't delete file !", 
                       PSP_MENU_WARNING_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
}

static void
psp_main_menu_cur_slot(int step)
{
  if (step == 1) {
    cur_slot++; if (cur_slot >= ATARI_MAX_SAVE_STATE) cur_slot = 0;
  } else if (step == -1) {
    cur_slot--; if (cur_slot < 0) cur_slot = ATARI_MAX_SAVE_STATE - 1;
  }
}

static void
psp_main_menu_screenshot(void)
{
  psp_screenshot_mode = 10;
}

static void
psp_main_menu_validate(void)
{
  /* Validate */
  ATARI.atari_snd_enable    = atari_snd_enable;
  ATARI.atari_render_mode   = atari_render_mode;
  ATARI.psp_cpu_clock       = psp_cpu_clock;
  ATARI.psp_reverse_analog  = psp_reverse_analog;
  ATARI.psp_skip_max_frame  = atari_skip_fps;
  ATARI.psp_skip_cur_frame  = 0;

  scePowerSetClockFrequency(ATARI.psp_cpu_clock, ATARI.psp_cpu_clock, ATARI.psp_cpu_clock/2);
}

static void
psp_main_menu_save_config()
{
  int error;

  psp_main_menu_validate();

  error = atari_save_configuration();

  if (! error) /* save OK */
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "File saved !", 
                       PSP_MENU_NOTE_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
  else 
  {
    psp_display_screen_menu();
    psp_sdl_fill_print(270,  80, "Can't save file !", 
                       PSP_MENU_WARNING_COLOR, PSP_MENU_BACKGROUND_COLOR);
    psp_sdl_flip();
    sleep(1);
  }
}

int
psp_main_menu_exit(void)
{
  SceCtrlData c;

  psp_display_screen_menu();
  psp_sdl_fill_print(270,  80, "press X to confirm !", 
                     PSP_MENU_WARNING_COLOR, PSP_MENU_BACKGROUND_COLOR);
  psp_sdl_flip();

  psp_kbd_wait_no_button();

  do
  {
    sceCtrlReadBufferPositive(&c, 1);

    if (c.Buttons & PSP_CTRL_CROSS) psp_sdl_exit(0);

  } while (c.Buttons == 0);

  psp_kbd_wait_no_button();
}

int 
psp_main_menu(void)
{
  long new_pad;
  long old_pad;
  int  last_time;
  int  end_menu;

  audio_pause();

  psp_kbd_wait_no_button();

  old_pad   = 0;
  last_time = 0;
  end_menu  = 0;

  atari_snd_enable     = ATARI.atari_snd_enable;
  atari_render_mode    = ATARI.atari_render_mode;
  atari_skip_fps       = ATARI.psp_skip_max_frame;
  psp_cpu_clock        = ATARI.psp_cpu_clock;
  psp_reverse_analog   = ATARI.psp_reverse_analog;

  while (! end_menu)
  {
    psp_display_screen_menu();
    psp_sdl_flip();


    SceCtrlData c;
    sceCtrlReadBufferPositive(&c, 1);

    new_pad = c.Buttons;

    if ((old_pad != new_pad) || ((c.TimeStamp - last_time) > PSP_MENU_MIN_TIME)) {
      last_time = c.TimeStamp;
      old_pad = new_pad;

    } else continue;

    if ((c.Buttons & (PSP_CTRL_LTRIGGER|PSP_CTRL_RTRIGGER|PSP_CTRL_START)) ==
        (PSP_CTRL_LTRIGGER|PSP_CTRL_RTRIGGER|PSP_CTRL_START)) {
      /* Exit ! */
      psp_sdl_exit(0);
    } else
    if ((c.Buttons & PSP_CTRL_RTRIGGER) == PSP_CTRL_RTRIGGER) {
      psp_main_menu_reset();
      end_menu = 1;
    } else
    if ((new_pad & PSP_CTRL_CROSS ) || 
        (new_pad & PSP_CTRL_CIRCLE) || 
        (new_pad & PSP_CTRL_RIGHT ) ||
        (new_pad & PSP_CTRL_LEFT  )) 
    {
      int step;

      if (new_pad & PSP_CTRL_LEFT)  step = -1;
      else 
      if (new_pad & PSP_CTRL_RIGHT) step =  1;
      else                          step =  0;

      switch (cur_menu_id ) 
      {
        case MENU_SOUND     : atari_snd_enable = ! atari_snd_enable;
        break;              
        case MENU_SKIP_FPS  : psp_main_menu_skip_fps( step );
        break;              
        case MENU_ANALOG    : psp_reverse_analog = ! psp_reverse_analog;
        break;              
        case MENU_RENDER    : psp_main_menu_render( step );
        break;              
        case MENU_CLOCK     : psp_main_menu_clock( step );
        break;
        case MENU_CUR_SLOT  : psp_main_menu_cur_slot(step);
        break;
        case MENU_LOAD_SLOT : if (step) psp_main_menu_cur_slot(step);
                              else      psp_main_menu_load_slot();
        break;
        case MENU_SAVE_SLOT : if (step) psp_main_menu_cur_slot(step);
                              else      psp_main_menu_save_slot();
        break;
        case MENU_DEL_SLOT  : if (step) psp_main_menu_cur_slot(step);
                              else      psp_main_menu_del_slot();
        break;
        case MENU_LOAD_ROM   : psp_main_menu_load(FMGR_FORMAT_ROM);
                               old_pad = new_pad = 0;
        break;              
        case MENU_KEYBOARD   : psp_keyboard_menu();
                               old_pad = new_pad = 0;
        break;
        case MENU_SCREENSHOT : psp_main_menu_screenshot();
                               end_menu = 1;
        break;              

        case MENU_SAVE_CFG  : psp_main_menu_save_config();
        break;
        case MENU_ORIGINAL  : psp_main_menu_original();
                              end_menu = 1;
        break;
        case MENU_RESET     : psp_main_menu_reset();
                              end_menu = 1;
        break;

        case MENU_BACK      : end_menu = 1;
        break;

        case MENU_EXIT      : psp_main_menu_exit();
        break;
      }

    } else
    if(new_pad & PSP_CTRL_UP) {

      if (cur_menu_id > 0) cur_menu_id--;
      else                 cur_menu_id = MAX_MENU_ITEM-1;

    } else
    if(new_pad & PSP_CTRL_DOWN) {

      if (cur_menu_id < (MAX_MENU_ITEM-1)) cur_menu_id++;
      else                                 cur_menu_id = 0;

    } else  
    if(new_pad & PSP_CTRL_SQUARE) {
      /* Cancel */
      end_menu = -1;
    } else 
    if(new_pad & PSP_CTRL_SELECT) {
      /* Back to ATARI */
      end_menu = 1;
    }
  }
 
  if (end_menu > 0) {
    psp_main_menu_validate();
  }

  psp_kbd_wait_no_button();

  psp_sdl_clear_screen( PSP_MENU_BLACK_COLOR );
  psp_sdl_flip();
  psp_sdl_clear_screen( PSP_MENU_BLACK_COLOR );
  psp_sdl_flip();

  audio_resume();

  return 1;
}

